<?php
	class CssStreamPlugin extends PluginHelperStream implements PluginLayoutOutputStream {
		/**
		 * The list of filenames which should be read
		 * 
		 * @var array
		 */
		protected $filelist = array();
		
		
		/**
		 * The name of this page
		 * 
		 * @var string
		 */
		protected $name = '';
		
		
		
		/**
		 * Constructor
		 * 
		 * @param Configuration $config
		 *        The current configuration object
		 * @param mixed         $loader
		 *        The object loading this plugin object
		 * @param string        $path
		 *        The path to open
		 * @param string        $mode
		 *        The file mode to open the path in
		 * @param array         $params
		 *        An array of loader specific stream properties
		 * 
		 * @throws NoStreamResourceError
		 */
		public function initialize($config, $loader, $path, $mode='r+', array $params=array()) {
			$this->config = $config;
			$this->params = $params;
			
			$url = parse_url($path);
			// Combine the host and path URL parts
			if(isset($url['host']) && isset($url['path'])) {
				$filelist = "{$url['host']}:{$url['path']}";
			} elseif(isset($url['host'])) {
				$filelist = $url['host'];
			} elseif(isset($url['path'])) {
				$filelist = $url['path'];
			} else {
				$filelist = null;
			}
			
			// Convert all slashes to colons to prevent anyone from using them
			// to open files in a different directory
			$filelist = str_replace("\\", ':', $filelist);
			$filelist = str_replace('/',  ':', $filelist);
			
			// Remove all colons at the start of the filelist
			while(substr($filelist, 0, 1) == ':') {
				$filelist = substr($filelist, 1);
			}
			// Remove all colons at the end of the filelist
			while(substr($filelist, -1) == ':') {
				$filelist = substr($filelist, 0, strlen($filelist)-1);
			}
			// Strip all duplicate solons in the filelist
			while(strpos($filelist, '::') !== false) {
				$filelist = str_replace('::', ':', $filelist);
			}
			
			$this->filelist = explode(':', $filelist);
			
			return $this;
		}
		
		
		/**
		 * Render and display the content
		 *
		 * @return bool Success?
		 */
		public function display() {
			if(!headers_sent()) {
				header('Content-Type', 'text/css');
			}
			
			$basedir="content://{$this->config->config()->fetch('directory')}";
			
			foreach($this->filelist as $filename) {
				// Check if the given filename can be read
				if(is_readable("{$basedir}/{$filename}")) {
					// Print headline
					vprintf("/********** %s **********/\n", $filename);
					
					// Gather the stylesheet contents
					$content = file_get_contents("{$basedir}/{$filename}");
					
					// Correct all stylesheet URLs
					preg_match_all(
						'/url\s*\(\s*([^)]+)\s*\)/', $content, $matches
					);
					foreach($matches[1] as $match) {
						// Generate the URL
						$url = UrlMain::get_url($match, $basedir);

						// Replace the URL in the content
						$content = str_replace($match, $url, $content);
					}
					
					// Compress stylsheet if requested
					if($this->config->config()->fetch('compress')) {
						$content = CssMain::compress($content);
					}
					
					// Escape special control sequences
					$content = str_replace('<$', '<$LT$>$', $content);
					
					// Add stylesheet contents
					print("{$content}\n\n");
					
					// Only flush if headers where already sent as calling the
					// flush()-function does this implicitaly
					if(headers_sent()) {
						flush();
					}
				} else {
					// Display the missing filename as an invalid construct
					// which will be displayed as an error in the browser
					print("ERROR: Could not read CSS file: {$filename};\n\n");
				}
			}
		}
		
		
		/**
		 * Does this stream exist, allways returns true as there will be
		 * induvitual messages for each non-existing file
		 * 
		 * @return bool
		 */
		public function exists() {
			return true;
		}
		
		
		/**
		 * Save all data that hasn't been saved yet, this is read-only so this
		 * method does nothing
		 */
		public function flush() {}
		
		
		/**
		 * Return the real filepath of the stream
		 * 
		 * @return string|false
		 */
		public function get_filepath() {
			$filepath = 'css://';
			foreach($this->filelist as $filename) {
				$filepath .= $filepath . ':';
			}
			
			return substr($filepath, 0, -1);
		}
		
		
		/**
		 * Return a string which uniquly identifies the resource loaded
		 * 
		 * @return string
		 */
		public function get_identifier() {
			return implode($this->filelist, ':');
		}
		
		
		/**
		 * Read $length characters starting at $start
		 * 
		 * If $length is null all remaining characters will be read.
		 * 
		 * @param integer [$start=0]
		 *        Where to start from reading
		 * @param integer [$length=ꝏ]
		 *        How many characters to read
		 * @return string|false
		 */
		public function read($start=0, $length=null) {}
		
		
		/**
		 * Render this page but do not emit any headers or content
		 * 
		 * @return string
		 */
		public function render() {
			// Enable output buffering
			ob_start();
			
			print("<\$HEADER 'Content-Type: text/css' $>");
			
			// "Display" the page into the output buffer
			$this->display();
			
			// Return buffer contents
			return ob_get_clean();
		}
		
		
		/**
		 * Collect and return the page's stat information
		 *
		 * @see http://php.net/manual/function.stat.php
		 * @return array
		 */
		public function stat() {
			return array();
		}
		
		
		/**
		 * Change the content of the page starting from position $start
		 * 
		 * @param integer [$start=0]
		 *        Where to start from writing
		 * @param string  $content
		 *        What to write
		 * @return bool Success?
		 */
		public function write($start=0, $content) {
			$this->buffering = true;
			
			if($start > 0) {
				$content = $this->read(0, $start) . $content . $this->read($start+strlen($content));
			}
			
			return $this->resource->write($content);
		}
	}
