<?php
	/**
	 * Display the requested page
	 *
	 * @package aCMS
	 */
	require_once(__DIR__ . '/' . "AOF/AOFError.class.php");
	
	require_once(__DIR__ . '/' . "initialize.inc.php");
	require_once(__DIR__ . '/' . "report.inc.php");
	require_once(__DIR__ . '/' . "classes/Configuration.class.php");
	require_once(__DIR__ . '/' . "RequestOutputStreamCache.class.php");
	require_once(__DIR__ . '/' . "classes/PluginLoader.class.php");
	
	
	
	/**
	 * Base exception class for all request related errors
	 */
	class RequestError extends AOFError {}
	
	/**
	 * Exception class called if a stream requested could not be found
	 */
	class StreamInvalidError extends RequestError {
		/**
		 * @param string $stream
		 *        The name of the stream which couldn't be loaded
		 */
		public function __construct($stream) {
			parent::__construct(
			  "The requested output stream \"{$stream}\" is invalid, not "  .
			  "defined or does not implement the PluginLayoutOutputStream " .
			  "interface."
			);
		}
	}
	
	/**
	 * Exception class called if the error stream used does not exist
	 */
	class ErrorResourceNotFoundError extends RequestError {
		/**
		 * A list of headers
		 *
		 * @var array
		 */
		protected $headers = array('HTTP/1.0 404 Not Found');
		
		
		/**
		 * @param string $stream
		 *        The name of the stream originally called
		 * @param string $resource
		 *        The name of the resource originally called
		 * @param string $error_stream
		 *        The name of the stream implementing the error page
		 * @param string $error_resource
		 *        The name of the resource implementing the error page
		 */
		public function __construct($stream, $resource, $error_stream, $error_resource) {
			parent::__construct(
			  "The requested resource \"{$stream}:{$resource}\" could not be " .
			  "be found additionally your error resource " .
			  "\"{$error_stream}:{$error_resource}\" could not be found."
			);
		}
	}
	
	
	
		
	/**
	 * 
	 */
	class Request extends AOFObject {
		/**
		 * The request parameters
		 * 
		 * @var array
		 */
		protected $params = array();
		
		/**
		 * The stream loaded to handle the request
		 * 
		 * @var mixed
		 */
		protected $stream;
		
		protected $config;
		
		
		
		/**
		 * @param array [$params=$_POST+$_GET]
		 *        An array containing the request parameters to display
		 */
		public function __construct(array $params, Configuration $config=null) {
			// Find parameters list
			$this->params = $params;
			// Find configuration
			if(is_object($config)) {
				$this->config = $config->main()->create_new("request.");
			} else {
				global $CONFIGURATION;
				$this->config = $CONFIGURATION->create_new("request.");
			}
			
			// Find the requested stream name
			if(isset($this->params['type'])) {
				$this->requested_stream = $this->params['type'];
			} else {
				$this->requested_stream =
						$this->config->config()->fetch('type.default');
			}
			// Find the requested resource name
			if(isset($this->params['resource'])) {
				$this->requested_resource = $this->params['resource'];
			} else {
				$this->requested_resource = '';
			}
			
			// Initialize requested stream with the requested resource
			$loader = new PluginLoader($this->requested_stream, $this->config);
			$this->stream = $loader->load(
				'stream',
				$this,
				$this->requested_resource,
				'r',
				array_merge($this->params, array(
						'called-as' => 'default',
						'original'  => null
				))
			);
			
			// Show invalid stream names as 404 errors
			if(!is_object($this->stream)) {
				return;
			}
			
			// Check if the stream is an output stream
			if(!self::is_output_stream($this->stream)) {
				throw new StreamInvalidError($this->requested_stream);
			}
			
			$this->register_streams();
		}
		
		
		public function register_streams() {
			foreach(PluginLoader::list_plugins('stream') as $name) {
				$loader = new PluginLoader($name);
				$loader->set_initializing(false);
				
				stream_wrapper_register($name, $loader->load('stream'));
			}
		}
		
		
		public static function is_output_stream($object) {
			if(!is_object($object)) {
				return false;
			}
			
			$reflection = new ReflectionObject($object);
			if(!$reflection->implementsInterface('PluginLayoutOutputStream')) {
				return false;
			}
			
			return true;
		}
		
		
		public function display() {
			if(is_object($this->stream) && $this->stream->exists()) {//TODO: Implement full-caching mode
				//if($this->stream->is_cacheable()) {
				if(true) {
					// Initialize cache handeling object
					$cache = new RequestOutputStreamCache(
							$this->requested_stream,
							$this->stream->get_identifier()
					);
					
					// Make sure there is a valid cache available
					if(!$cache->cache_is_valid()) {
						$content = $this->stream->render();
						$cache->process_content($content);
					}
					
					// Execute the cache file
					$cache->cache_execute();
				} else {
					$this->stream->display();
				}
			} else {
				// Send 404 error: Not found
				header("HTTP/1.0 404 Not found");
				
				// Initialize 404-error page stream with the default 404-error
				// page resource and pass the original stream called
				$loader = new PluginLoader(
						$this->config->config()->fetch('error404.stream'),
						$this->config
				);
				$error_stream = $loader->load(
					'stream',
					$this,
					$this->config->config()->fetch('error404.resource'),
					'r',
					array_merge($this->params, array(
						'called-as' => 'error404',
						'original'  => $this->stream
					))
				);
				
				// Check if the stream is designed as an output stream
				if(is_object($error_stream)
				&& !self::is_output_stream($error_stream)) {
					throw new StreamInvalidError(
							$this->config->config()->fetch('error404.stream')
					);
				}
				
				// Display error stream if it exists
				if($error_stream->exists()) {
					$error_stream->display();
				// Throw an exception if not
				} else {
					throw new ErrorResourceNotFoundError(
							$this->requested_stream,
							$this->requested_resource,
							$this->config->config()->fetch('error404.stream'),
							$this->config->config()->fetch('error404.resource')
					);
				}
			}
		}
	}
	
	
	
	
	/**
	 * The plugin layout for all resource plugins
	 */
	interface PluginLayoutResource extends PluginLayout {
		/**
		 * @param Configuration $config
		 *        The current configuration
		 * @param mixed         $loader
		 *        The object loading this plugin
		 * @param string        $location
		 *        The location/name of the resource to open
		 */
		public function __construct($config, $loader, $location);
		
		/**
		 * Return if the resource requested actually exists
		 * 
		 * @return bool
		 */
		public function exists();
		
		/**
		 * Flush any remaining data still not saved
		 */
		//public function flush();
		
		/**
		 * Return the length of the content of the current resource
		 * 
		 * @return integer
		 */
		public function get_length();
		
		/**
		 * Return the real filepath actually opened by this resource or null if
		 * there is none
		 * 
		 * @return string|null
		 */
		public function get_real_path();
		
		/**
		 * Return the PHP resource handler actually used to perform the
		 * operations which are requested
		 * 
		 * The returned PHP resource of this function may be used to generate
		 * stat-information on this resource object if {@see self::stat()} is
		 * not defined.
		 * @see http://php.net/manual/function.fstat.php
		 * 
		 * @return resource|null
		 */
		//public function get_resource();
		
		/**
		 * Read $length characters starting at $start
		 * 
		 * If $length is null all remaining characters will be read.
		 * 
		 * @param integer [$start=0]
		 *        Where to start from reading
		 * @param integer [$length=ꝏ]
		 *        How many characters to read
		 * @return string|false
		 */
		public function read($start=0, $length=null);
		
		/**
		 * Generate and return the stat information of this resource (optional)
		 * 
		 * If the stat information returned is incomplete than it should be
		 * completed by the {@see PluginHelperStream::stat_generator} function.
		 * 
		 * @see http://php.net/manual/function.stat.php
		 * @return array
		 */
		//public function stat();
		
		/**
		 * Change the content of the resource starting at position $start
		 * 
		 * @param integer [$start=0]
		 *        Where to start from writing
		 * @param string  $content
		 *        What to write
		 * @return bool Success?
		 */
		public function write($start=0, $content);
	}
	
	
	
	/**
	 * The layout for stream plugins
	 * 
	 * For compatibility reasons with stream_wrapper_register() initialize()
	 * instead of __construct() is used as constructer.
	 * 
	 * Stream plugins should extend {@see PluginHelperStream} since it makes
	 * live a lot easier, however this is not required. For more information
	 * checkout the documentation of the helper class.
	 */
	interface PluginLayoutStream extends PluginLayout {
		/**
		 * Constructor
		 * 
		 * @param Configuration $config
		 *        The current configuration object
		 * @param mixed         $loader
		 *        The object loading this one
		 * @param string        $path
		 *        The path to open
		 * @param string        $mode
		 *        The file mode to open the path in
		 * @param array         $params
		 *        An array of loader specific stream properties
		 *        (e.g.: "language")
		 */
		public function initialize($config, $loader, $path, $mode='r+', array $params=array());
		
		/**
		 * Return if the stream actually exists
		 * 
		 * @return bool
		 */
		public function exists();
		
		/**
		 * Save all data that hasn't been saved yet
		 */
		public function flush();
		
		/**
		 * Generate a secondary URL under which this stream may be reached
		 * 
		 * @return string|false
		 */
		//public function generate_url();
		
		/**
		 * Return the real filepath of the stream or null if there is none
		 * 
		 * @return string|null
		 */
		public function get_filepath();
		
		/**
		 * Return a string which uniquly identifies the resource loaded
		 * 
		 * @return string
		 */
		public function get_identifier();
		
		/**
		 * Return the resource object actually used
		 * 
		 * @return resource|null
		 */
		//public function get_resource();
		
		/**
		 * Read $length characters starting at $start
		 * 
		 * If $length is null all remaining characters will be read.
		 * 
		 * @param integer [$start=0]
		 *        Where to start from reading
		 * @param integer [$length=]
		 *        How many characters to read
		 * @return string|null
		 */
		public function read($start=0, $length=null);
		
		/**
		 * Change the content of the stream starting from position $start
		 * 
		 * @param integer [$start=0]
		 *        Where to start from writing
		 * @param string   $content
		 *        What to write
		 * @return bool Success?
		 */
		public function write($start=0, $content);
		
		/**
		 * Collect and return the stream's stat information
		 *
		 * @see http://php.net/manual/function.stat.php
		 * @return array
		 */
		public function stat();
		
		
		
		/**
		 * Return the resource actually used for the stream data
		 * 
		 * @param integer $cast_as
		 *        A STREAM_CAST_* constant defining why the cast was made
		 * @return resource|false
		 */
		public function stream_cast($cast_as);
		
		/**
		 * Close the stream and flush its contents
		 * 
		 * @internal
		 */
		public function stream_close();
		
		/**
		 * Return if the end of the stream was reached
		 * 
		 * @internal
		 * @return bool Was the stream's end reached?
		 */
		public function stream_eof();
		
		/**
		 * Flush the contents of the stream
		 * 
		 * @internal
		 * @return bool
		 */
		public function stream_flush();
		
		//TODO: public function stream_lock($operation);
		
		//TODO: public function stream_metadata($path, $option, $var);
		
		/**
		 * The PHP stream initializer
		 * 
		 * @internal
		 * @param  string  $path    The path to open
		 * @param  string  $mode    The mode in which the URL should be opened
		 * @param  integer $options STREAM_* parameter constants
		 * @param &string  $opened  The path which was actually opened
		 */
		public function stream_open($path, $mode, $options, &$opened);
		
		/**
		 * Read $count bytes from the current position and return them
		 * 
		 * @internal
		 * @param integer $count
		 *        How many characters to read
		 * @return string
		 */
		public function stream_read($count);
		
		/**
		 * Change the position of the stream pointer
		 * 
		 * @internal
		 * @param integer $offset
		 *        How far to move the pointer
		 * @param integer $whence
		 *        A SEEK_* constant specifying the action
		 * 
		 * @return bool In range?
		 */
		public function stream_seek($offset, $whence);
		
		//TODO: public function stream_set_option($option, $arg1, $arg2);
		
		/**
		 * Return the stat information of the stream
		 * 
		 * @internal
		 * @see http://php.net/manual/function.stat.php
		 * @return array
		 */
		public function stream_stat();
		
		/**
		 * Return the current position of the stream pointer
		 * 
		 * @internal
		 * @return integer
		 */
		public function stream_tell();
		
		/**
		 * Write data at the current stream position and update stream pointer
		 * if successful
		 * 
		 * @internal
		 * @param string $data
		 *        The data to write into the file
		 * @return integer|false How many bytes were written?
		 */
		public function stream_write($data);
		
		/**
		 * Stat the contents of a stream which wasn't previously opened
		 * 
		 * @param $path
		 *        The full URL on which stat() should be called on
		 * @param $flags
		 *        An OR-ed list of STREAM_URL_STAT_*-constants which should be
		 *        used for processing
		 */
		static public function url_stat($path, $flags);
	}
	
	
	/**
	 * Extension of the normal stream which also allows the stream to be
	 * displayed as a webpage
	 */
	interface PluginLayoutOutputStream extends PluginLayoutStream {
		/**
		 * Display a rendered version of this stream as a webpage
		 * 
		 * @return bool Success?
		 */
		public function display();
		
		/**
		 * Render this stream as a webpage and return the generated content
		 * 
		 * @return string|null
		 */
		public function render();
	}
	
	
	
	
	/**
	 * Abstract helper class for stream plugins
	 * 
	 * This class handles all the streaming operations of the `streamwrapper`
	 * prototype class and provides a more user-friendly front- end.
	 * 
	 * All variables and function with the @internal tag are not meant for user
	 * useage instead they are only intended .
	 * 
	 * @see http://php.net/manual/class.streamwrapper.php
	 */
	abstract class PluginHelperStream implements PluginLayoutStream {
		/**
		 * The currently opened page (if called as stream wrapper)
		 * 
		 * @internal
		 * @var Page
		 */
		public static $config_static;
		
		/**
		 * The loading object loading this stream
		 * 
		 * @internal
		 * @var mixed
		 */
		public static $loader_static;
		
		/**
		 * The current position of the file pointer
		 * 
		 * @internal
		 * @var integer
		 */
		private $position = 0;
		
		/**
		 * Is there any buffered content that has to be saved?
		 * 
		 * If this flag is set to true the next time the stream is flushed or
		 * if the stream is closed the flush() method will be called.
		 * 
		 * @var bool
		 */
		protected $buffering = false;
		
		/**
		 * The length of the opened stream
		 * 
		 * @var integer
		 */
		protected $length = 0;
		
		
		
		/**
		 * Closing function
		 */
		public function close() {
			$this->stream_flush();
		}
		
		
		/**
		 * Close stream upon destruction
		 */
		public function __destruct() {
			$this->close();
		}
		
		
		
		/**
		 * A function that processes an array of stat entries and fills in all
		 * missing entries
		 * 
		 * At first this function will fill in all missing keyword entries
		 * within you array with default values (usually 0 or -1!) then all
		 * known keyword will be enumerated and added to the stat array as well.
		 * 
		 * @see http://php.net/manual/function.stat.php
		 * @param array [$keywords=array()]
		 *        A list of stat entries already set
		 * 
		 * @return array A PHP stat compatible array
		 */
		public static function stat_generator(array $keywords=array()) {
			// Generate array of default keyword items
			$default = array(
			  'dev'    =>  0, 'ino'   =>  0, 'mode'    =>  0,
			  'nlink'  =>  0, 'uid'   =>  0, 'gid'     =>  0,
			  'rdev'   =>  0, 'size'  =>  0, 'atime'   => -1,
			  'mtime'  => -1, 'ctime' => -1, 'blksize' =>  0,
			  'blocks' => -1
			);
			
			// Merge default and supplied keyword data (The supplied data will
			// have a higher presence than the default one)
			$data = array_merge($default, $keywords);
						
			// Generate enumerated array
			$result     = array();
			$result[0]  = $data['dev'];  $result[1]  = $data['ino'];
			$result[2]  = $data['mode']; $result[3]  = $data['nlink'];
			$result[4]  = $data['uid'];  $result[5]  = $data['gid'];
			$result[6]  = $data['rdev']; $result[7]  = $data['size'];
			$result[8]  = $data['atime'];$result[9]  = $data['mtime'];
			$result[10] = $data['ctime'];$result[11] = $data['blksize'];
			$result[12] = $data['blocks']; 
			// Merge the enumerated stat array with the keyword-based one
			$result = array_merge($result, $data);
			
			return $result;
		}
		
		
		
		/**
		 * Return the resource actually used for the stream data
		 * 
		 * @param integer $cast_as
		 *        A STREAM_CAST_* constant defining why the cast was made
		 * @return resource|false
		 */
		public function stream_cast($cast_as) {
			if(function_exists(array($this, 'get_resource'))) {
				return $this->get_resource();
			} else {
				return false;
			}
		}
		
		
		/**
		 * Close the stream and flush its contents
		 * 
		 * @internal
		 */
		public function stream_close() {
			$this->close();
		}
		
		
		/**
		 * Return if the end of the stream was reached
		 * 
		 * @internal
		 * @return bool Was the stream's end reached?
		 */
		public function stream_eof() {
			return $this->position >= $this->length;
		}
		
		
		/**
		 * Flush the contents of the stream
		 * 
		 * @internal
		 * @return bool
		 */
		public function stream_flush() {
			// Flush stream data if there is anything buffered
			if($this->buffering) {
				$this->buffering = false;
				return $this->flush();
			} else {
				return true;
			}
		}
		
		
		//TODO: public function stream_lock($operation)
		
		
		/**
		 * The PHP stream initializer
		 * 
		 * @internal
		 * @param  string  $path    The path to open
		 * @param  string  $mode    The mode in which the URL should be opened
		 * @param  integer $options STREAM_* parameter constants
		 * @param &string  $opened  The path which was actually opened
		 */
		public function stream_open($path, $mode, $options, &$opened) {
			// Parse the URL to open
			$url = @parse_url($path);
			// Check if the URL is valid
			if(!$url) {
				// Report an error if requested and return
				if($options == ($options|STREAM_REPORT_ERRORS)) {
					report(WARNING, 'stream', "Invalid path: {$path}");
				}
				return false;
			}
			
			// Make sure we have the configuration object available
			global $CONFIGURATION;
			$config = $CONFIGURATION->create_new("{$url['scheme']}.");
			
			// Initialize the current object
			$this->initialize(
				$config,
				self::$loader_static,
				$path,
				$mode,
				array("use_include_path" => ($options == ($options|STREAM_USE_PATH)))
			);
			
			// Check if initialization was successful
			if($this->exists()) {
				// Set the real filepath
				$opened = $this->get_filepath();
				
				return true;
			} else {
				// Report an error if requested and return
				if($options == ($options|STREAM_REPORT_ERRORS)) {
					report(WARNING, 'stream',
                    	"Could not initialize stream: {$url['scheme']}");
				}
				return false;
			}
		}
		
		
		/**
		 * Read $count bytes from the current position and return them
		 * 
		 * @internal
		 * @param integer $count
		 *        How many characters to read
		 * @return string
		 */
		public function stream_read($count) {
			// Save the current stream position
			$current = $this->position;
			
			// Update current stream position and return the content if we are
			// within the file boundries
			if($this->stream_seek($count, SEEK_CUR)) {
				// 
				return $this->read($current, $count);
			} else {
				return '';
			}
		}
		
		
		/**
		 * Change the position of the stream pointer
		 * 
		 * @internal
		 * @param integer $offset
		 *        How far to move the pointer
		 * @param integer $whence
		 *        A SEEK_* constant specifying the action
		 * 
		 * @return bool In range?
		 */
		public function stream_seek($offset, $whence) {
			switch($whence) {
				case SEEK_SET:
					if($offset >= 0 && $this->length > $offset) {
						$this->position = $offset;
						return true;
					} else {
						return false;
					}
				break;
				case SEEK_CUR:
					if($offset >= 0) {
						$this->position += $offset;
						return true;
					} else {
						return false;
					}
				break;
				case SEEK_END:
					if($this->length + $offset >= 0) {
						$this->position = $this->length + $offset;
						return true;
					} else {
						return false;
					}
				break;
				default:
					return false;
				break;
			}
		}
		
		
		//TODO: public function stream_set_option($option, $arg1, $arg2)
		
		
		/**
		 * Return the stat information of the stream
		 * 
		 * @internal
		 * @see http://php.net/manual/function.stat.php
		 * @return array
		 */
		public function stream_stat() {
			return $this->stat();
		}
		
		
		/**
		 * Return the current position of the stream pointer
		 * 
		 * @internal
		 * @return integer
		 */
		public function stream_tell() {
			return $this->position;
		}
		
		
		/**
		 * Write data at the current stream position and update stream pointer
		 * if successful
		 * 
		 * @internal
		 * @param string $data
		 *        The data to write into the file
		 * @return integer|false How many bytes were written?
		 */
		public function stream_write($data) {
			// Data was written so it has to be flushed at some point
			$this->buffering = true;
			
			// Write data and update stream pointer position if successful
			if($this->write($this->position, $data)) {
				$this->position += strlen($data);
				return strlen($data);
			} else {
				return false;
			}
		}
		
		
		/**
		 * Stat the contents of a stream which wasn't previously opened
		 * 
		 * @param $path
		 *        The full URL on which stat() should be called on
		 * @param $flags
		 *        An OR-ed list of STREAM_URL_STAT_*-constants which should be
		 *        used for processing
		 */
		static public function url_stat($path, $flags) {
			// Initialize new instance of this class
			$self = new static();
			
			// Use stream_open for initialization as it provides a very similar
			// interface to the one implemented here
			$self->stream_open($path, 'r', $flags, $filepath);
			
			// Store the results of the stat operation done on the stream
			$stat = $self->stat();
			
			// Close the current stream
			$self->stream_close();
			
			// Return the stat results
			return $stat;
		}
	}
	
	
	
	

