<?php
	/**
	 * Collection of functions for error reporting
	 *
	 * @package aCMS
	 */
	require_once(__DIR__ . '/' . "classes/Configuration.class.php");
	
	require_once(__DIR__ . '/' . "AOF/AOFObject.class.php");
	require_once(__DIR__ . '/' . "AOF/AOFError.class.php");
	
	
	class PHPException extends Exception {
		function __construct($string, $filename, $lineno) {
			$text  = "<br />\n";
			$text .= "<strong>PHP has reported an critical error:</strong><br />\n";
			$text .= "<em>$string</em> (in $filename at $lineno)<br />\n";
			
			parent::__construct($text);
		}
	}
	
	
	
	/**
	 * A class allowing you to display and log a report message
	 */
	class ReportMessage extends AOFObject {
		/**
		 * An array of logging levels where the key is the level name and the
		 * value is the level's importance as an iteger
		 * 
		 * array("{$name}" => (int) $importance, ...)
		 * 
		 * @var array
		 */
		static protected $levels = array();
		
		
		/**
		 * A cache of all configuration values due to massive amouts of query
		 * through report2
		 * 
		 * @var array
		 */
		static protected $config_data = array(
			'config'  => null,
			'runtime' => null
		);




		/**
		 * @var Configuration
		 */
		protected $config;
		
		
		
		
		/**
		 * Register a new report importance level
		 */
		static public function level_register($name, $importance) {
			// Register the report level
			self::$levels[strtolower($name)] = $importance;
			
			// Define a constant for the level
			define(strtoupper($name), $importance);
		}
		
		
		/**
		 * Find an level's importance and name from just one of the two given
		 *
		 * @param string|integer $level
		 *        Error level's code or name
		 * 
		 * @return array array(int) $importance, "{$name}")
		 */
		static public function level_find($level) {
			// If $level is an integer then we'll try to find it inside of the
			// flipped self::$levels array (This means that the importance of
			// the level is the key of the array)
			if(is_integer($level)) {
				$level_int = $level;
				
				$importances = array_flip(self::$levels);
				if(isset($importances[$level_int])) {
					$level_str = $importances[$level_int];
				} else {
					$level_str = 'unknown';
				}
			// If $level is string we'll just try to find inside the normal
			// self::$levels array
			} elseif(is_string($level)) {
				$level_str = strtolower($level);
				
				if(isset(self::$levels[$level_str])) {
					$level_int = self::$levels[$level_str];
				} else {
					$level_int = UNKNOWN;
				}
			// If the level is neither an integer nor a string than we give up
			} else {
				$level_int = UNKNOWN;
				$level_str = 'unknown';
			}
			
			return array($level_int, $level_str);
		}
		
		
		
		
		/**
		 * @param string|integer $level     The level name or code
		 * @param string         $component The name of the component logging
		 * @param string         $message   The message to log
		 * @param array          $items     Additional information
		 * @param integer        $tb_skip   How many traceback items to skip
		 */
		public function __construct($level, $component, $message, $data=NULL, $tb_skip=1) {
			$this->level     = $level;
			$this->component = $component;
			$this->message   = $message;
			$this->data      = $data;
			$this->tb_skip   = $tb_skip;
			
			global $CONFIGURATION;
			if(is_object($CONFIGURATION)) {
				if(empty(self::$config_data['config'])) {
					$config = $CONFIGURATION->create_new('report.')->config();
					self::$config_data['config'] = array(
						'output.hide'      => $config->fetch('output.hide'),
						'output.level'     => $config->fetch('output.level'),
						'output.traceback' => $config->fetch('output.traceback'),
						
						'logfile.hide'      => $config->fetch('logfile.hide'),
						'logfile.level'     => $config->fetch('logfile.level'),
						'logfile.directory' => $config->fetch('logfile.directory'),
					);
				}
				
				if(empty(self::$config_data['runtime'])) {
					self::$config_data['runtime'] = array(
						'output.disabled'  => false,
						'logfile.disabled' => false,
					);
				}
			}
		}
		
		
		/**
		 * Check logging file path environment and return the path of a 
		 * writable logging file path
		 * 
		 * @return string|null
		 */
		protected function generate_logfile_path() {
			if(is_null(self::$config_data['config'])
			|| is_null(self::$config_data['runtime'])) {
				return null;
			}
			
			// Check if the logging directory is writable
			$directory = self::$config_data['config']['logfile.directory'];
			if(!is_writeable($directory)){
				// Deactivate logging to not repeat the following warning
				// every time `report` is called
				self::$config_data['runtime']['logfile.disabled'] = true;
				
				// Display a warning
				$this->to_output(WARNING, 'report',
				  "Logging directory \"{$directory}\" is not writeable!"
				);
				
				return null;
			}
			
			// Generate the logfile path based on the current date
			$logfile = $directory.'/'.date('Y-m-d').'.log';
			
			if(!file_exists($logfile)) {
				// Create logging file
				file_put_contents($logfile, '');
			} elseif(!is_writeable($logfile)) {
				// Deactivate logging to not repeat the following warning
				// every time a message is reported
				self::$config_data['runtime']['logfile.disabled'] = true;
				
				// Display a warning
				$this->to_output(WARNING, 'report',
				  "Logging file \"{$logfile}}\" is not writeable!"
				);
				
				return null;
			}
			
			return $logfile;
		}
		
		
		/**
		 * Check if a certain <em>report.{$cfgsec}.*</em> prefix is allowed
		 * to process the current message
		 * 
		 * @return bool
		 */
		public function config_check_if_allowed($cfgsec) {
			list($level_int, $level_str) = self::level_find($this->level);
			
			// Do test based on configuration settings if any where given
			if(is_array(self::$config_data['config'])
			&& is_array(self::$config_data['runtime'])) {
				// Check if this logging method was disabled
				if(isset(self::$config_data['runtime']["{$cfgsec}.disabled"])
				&& self::$config_data['runtime']["{$cfgsec}.disabled"]) {
					return false;
				}
				
				// Check if logging for this component was disabled
				$is_hidden = in_array(
					$this->component,
					self::$config_data['config']["{$cfgsec}.hide"]
				);
				if($is_hidden){
					return false;
				}
				
				// Check if the level used is lower than the minimum
				// logging level
				$minimum = self::level_find(
					self::$config_data['config']["{$cfgsec}.level"]
				);
				if($level_int < $minimum[0]) {
					return false;
				}
			} else {
				// Don't check for disabled components
				
				// If the message checked for is intented for displaying and
				// meets the required minimum logging level then allow this
				// action
				if($cfgsec == 'output'
				&& defined("REPORT_PRECONFIG_MINIMUM_DISPLAY_LEVEL")) {
					$minimum = self::level_find(
						REPORT_PRECONFIG_MINIMUM_DISPLAY_LEVEL
					);
					if($level_int >= $minimum[0]) {
						return true;
					}
				}
				
				return false;
			}
			
			return true;
		}
		
		
		/**
		 * 
		 */
		public function to_logfile() {
			// Check if logging of this message is allowed
			if(!$this->config_check_if_allowed('logfile')) {
				return true;
			}
			
			// Generate logging file path
			if(($logfile=$this->generate_logfile_path()) === NULL) {
				return false;
			}
			
			list($level_int, $level_str) = self::level_find($this->level);
			
			// Create the logging line
			// [date()] $level_str: $component: $message ($filename:$line)
			$line  = '[' . date('D M d H:i:s Y') . ']' . " ";
			$line .= "{$level_str}:"                   . " ";
			$line .= "{$this->component}:"             . " ";
			$line .= strip_tags($this->message);
			if(isset($this->data['file']) && isset($this->data['line'])) {
				$line .= " " . "({$this->data['file']}:{$this->data['line']})";
			}
			
			// Write the line to the logging file
			file_put_contents($logfile, "$line\n", FILE_APPEND);
			
			return true;
		}
		
		
		public function to_output() {
			// Check if displaying of this message is allowed
			if(!$this->config_check_if_allowed('output')) {
				return true;
			}
			
			list($level_int, $level_str) = self::level_find($this->level);
			
			// Generate basic HTML message
			$html  = "<div class=\"report_message\">\n";
			$html .= "\t<strong>{$level_str}</strong>:" . ' ';
			$html .=   "<em>{$this->component}:</em>"   . ' ';
			$html .=   "{$this->message}\n";
			
			// If any $items are given
			if(is_array($this->data) && count($this->data) > 0) {
				$html .= "\t<pre>\n";
				
				// Loop through the $items to find the longest $name
				$name_maxlen = array_reduce(array_keys($this->data),
				  create_function('$last, $name', '
				  	return (strlen($name) > $last) ? strlen($name) : $last;
				  '), 0
				);
				
				// Loop again through the $items and display them
				foreach($this->data as $name => $value) {
					// Define how much space is left after the colon
					$space = str_repeat(' ', $name_maxlen - strlen($name));
					
					$html .= "\t\t" . htmlspecialchars(
					  ucfirst($name) . ':' . $space . ' ' . $value
					) . "\n";
				}
				
				$html .= "\t</pre>\n";
			}
			
			// 
			if((is_null(self::$config_data['config'])
			 && REPORT_PRECONFIG_TRACEBACK)
			|| (is_array(self::$config_data['config'])
			 && self::$config_data['config']['output.traceback'])) {
				$backtrace = array_reverse(
				  array_slice(debug_backtrace(), $this->tb_skip)
				);
				
				$html .= "\t<p>Traceback:\n";
				$html .= "\t" . AOFError::create_trace_html($backtrace, false) . "\n";
				$html .= "\t</p>\n";
			}
			
			$html .= "</div>\n";
			
			print($html);
			flush();
		}
	}
	
	
	/**
	 * @constant integer UNKOWN -1
	 * @constant integer DEBUG   10
	 * @constant integer INFO    20
	 * @constant integer WARN    30
	 * @constant integer WARNING 30
	 */
	ReportMessage::level_register('unknown', -1);
	ReportMessage::level_register('debug',   10);
	ReportMessage::level_register('info',    20);
	ReportMessage::level_register('warn',    30);
	ReportMessage::level_register('warning', 30);
	
	
	
	
	/**
	 * Report a logging info
	 *
	 * @param string|integer $level     The level name or code
	 * @param string         $component The name of the component logging
	 * @param string         $message   The message to log
	 * @param array          $data      Additional information
	 * @param integer        $tb_skip   How many traceback items to skip
	 */
	function report($level, $component, $message, $data=NULL, $tb_skip=1) {
		$message =
			new ReportMessage($level, $component, $message, $data, $tb_skip);
		$message->to_logfile();
		$message->to_output();
		
		return true;
	}
	
	
	/**
	 * Log a message from the current scope
	 * 
	 * @param string|integer $level
	 *        The importance of the message
	 * @param string         $message
	 *        The message which should be logged
	 */
	function report2($level, $message) {
		// Create new report message object
		$message = new ReportMessage($level, "", $message, array(), 1);
		
		// Stop if there is not going to be any logging
		if(!$message->config_check_if_allowed('output')
		&& !$message->config_check_if_allowed('logfile')) {
			return;
		}
		
		// Generate and set the component name from stacktrace
		if(version_compare(PHP_VERSION, '5.4.0') >= 0) {
			// Limit the number of stack frames returned (added in PHP 5.4)
			$frames = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 2);
		} elseif(version_compare(PHP_VERSION, '5.3.6') >= 0) {
			$frames = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS);
		} else {
			$frames = debug_backtrace(false);
		}
		// Add class name and operator first if it was an object or class which
		// called this function
		if($frames[1]['type']) {
			$message->component = $frames[1]['class'] . $frames[1]['type'];
		}
		// Add function name of the caller
		$message->component .= $frames[1]['function'];
		
		// Populate the message
		$message->to_logfile();
		$message->to_output();
	}
	
	
	/**
	 * Wrap around PHP errors and convert them to our logging and error system
	 *
	 * @see http://www.php.net/manual/function.set-error-handler.php
	 * @param integer $type     Contains the PHP error code
	 * @param string  $string   Contains the PHP error message
	 * @param string  $filename Contains the filename of the error
	 * @param integer $lineno   Contains the line number of the error
	 * @access private
	 */
	function report_php_handler($type, $string, $filename=NULL, $lineno=NULL) {
		global $CONFIGURATION;
		
		// Make sure all PHP error constants exist
		if(!defined('E_STRICT')) {
			define('E_STRICT', 2048);
		}
		if(!defined('E_RECOVERABLE_ERROR')) {
			define('E_RECOVERABLE_ERROR', 4096);
		}
		if(!defined('E_DEPRECATED')) {
			define('E_DEPRECATED', 8192);
		}
		if(!defined('E_USER_DEPRECATED')) {
			define('E_USER_DEPRECATED', 16384);
		}
		
		// Find the loglevel to use
		switch($type) {
			case E_STRICT:
			case E_DEPRECATED:
			case E_USER_DEPRECATED:
				$level = INFO;
			case E_NOTICE:
			case E_USER_NOTICE:
			case E_WARNING:
			case E_USER_WARNING:
			case E_COMPILE_WARNING:
			case E_CORE_WARNING:
				$level = WARNING;
			break;
			
			case E_ERROR:
			case E_USER_ERROR:
			case E_RECOVERABLE_ERROR:
				throw new PHPException($string, $filename, $lineno);
			break;
		}
		
		// Decide if verbose mode we are in verbose mode
		if(defined("REPORT_DEBUGING_ENABLED")
		||(is_object($CONFIGURATION)
		&& $CONFIGURATION->config()->exists("report.php.show_silenced")
		&& $CONFIGURATION->config()->fetch("report.php.show_silenced"))) {
			$verbose = true;
		} else {
			$verbose = false;
		}
		
		// Create additional data array
		$data = array('File' => $filename, 'Line' => $lineno);
		
		// Create report message array
		$message =
		  new ReportMessage($level, 'php', $string, $data, 2);
		
		// Just log silenced messages if we aren't in verbose mode
		if(ini_get('error_reporting') == 0) {
			if($verbose) {
				$message->to_logfile();
			}
		} else {
			$message->to_logfile();
			$message->to_output();
		}
	}
	set_error_handler('report_php_handler');
	error_reporting(E_ALL | E_STRICT);
