<?php
	require_component("AOF/AOFError.class.php");
	require_component("classes/PluginLoader.class.php");
	
	
	/**
	 * Base class for exceptions in the Page-class
	 **/
	class PageError extends AOFError {
		/**
		 * The title of the error page
		 * @var string
		 */
		protected $title = 'The requested page can not be displayed';
	}
	
	/**
	 * Thrown if no page handler was found
	 */
	class PageRendererNotFoundError extends PageError {
		/**
		 * @param string $name The page's name
		 * @param string $type The page's original type
		 */
		public function __construct($name, $type) {
			$this->message  = "Sorry the requested page \"{$name}\" could ";
			$this->message .= "not be displayed because the invalid/missing ";
			$this->message .= "page renderer \"{$type}\" was used.";
		}
	}
	
	/**
	 * Exception class for errors if the error page could not be found
	 */
	class ErrorPageNotFoundError extends PageError {
		/**
		 * A list of headers
		 *
		 * @var array
		 */
		protected $headers = array('HTTP/1.0 404 Not Found');
		
		
		
		/**
		 * @param string $name The page's name
		 * @param string $type The page's original type
		 */
		public function __construct($name, $errorpage='') {
			$this->message  = "Sorry the requested page \"{$name}\" can not ";
			$this->message .= "be found, additionally your default error page ";
			$this->message .= "{$errorpage} is undefined.";
		}
	}
	
	/**
	 * Exception for resource errors while loading page
	 * 
	 * This exception gets thrown if a stream non-existing stream resource was
	 * set for loading pages
	 */
	class PageStreamResourceError extends PageError {
		/**
		 * @param string $name The non-existing resource
		 */
		public function __construct($name='') {
			if($name) {
				$this->message = "The stream resource \"$name\" does not exist";
			} else {
				$this->message = "No stream resource was set";
			}
			
			$this->solutions[] = "Make sure the resource exists";
			$this->solutions[] =
				"Change the configuration key <em>page.resource</em> to an " .
				"existing resource";
		}
	}
	
	
	
	
	/**
	 * Stream handling class for the page plugin
	 */
	class PageStreamPlugin extends PluginHelperStream implements PluginLayoutOutputStream {
		/**
		 * A list of cached headers used by the resource
		 * 
		 * This variable will always be empty if the resource supports the
		 * get_header and set_header methods.
		 * 
		 * @var array
		 */
		protected $header_cache;
		
		/**
		 * The name of this page
		 */
		protected $name = '';
		
		/**
		 * The language of this page
		 */
		protected $language = '';
		
		/**
		 * The loaded resource object
		 * 
		 * @var mixed
		 */
		protected $resource;
		
		
		
		/**
		 * Constructor
		 * 
		 * @param Configuration $config
		 *        The current configuration object
		 * @param mixed         $loader
		 *        The object loading this one
		 * @param string        $path
		 *        The path to open
		 * @param string        $mode
		 *        The file mode to open the path in
		 * @param array         $params
		 *        An array of loader specific stream properties
		 *        (e.g.: "language")
		 * 
		 * @throws NoStreamResourceError
		 */
		public function initialize($config, $loader, $path, $mode='r+', array $params=array()) {
			// Load automatically generated region data
			static $country_codes_translation_table;
			static $language_codes_translation_table;
			if(!is_array($country_codes_translation_table)
			|| !is_array($language_codes_translation_table)) {
				require(dirname(__FILE__) . '/' . "regiondata.thin.php");
			}
			
			$this->config = $config;
			$this->params = $params;
			
			if(!$config->runtime()->exists('stream')) {
				$config->runtime()->create('stream', $this);
			}
			$url = parse_url($path);
			
			// Check if path has more than one part (host/path)
			if(isset($url['host']) && isset($url['path'])) {
				$this->name = trim("{$url['host']}/{$url['path']}");
			} elseif(isset($url['path'])) {
				$this->name = trim($url['path']);
			} elseif(isset($url['host'])) {
				$this->name = trim($url['host']);
			} else {
				$this->name = '';
			}
			
			$this->name = str_replace('\\', '/', $this->name);
			
			// Cut off all slashes at the beginning of the page name
			while(substr($this->name, 0, 1) == '/') {
				$this->name = substr($this->name, 1);
			}
			// Cut off all slashes at the end of the page name
			while(substr($this->name, -1) == '/') {
				$this->name = substr($this->name, 0, strlen($this->name)-1);
			}
			// Strip all double slashes
			while(strpos($this->name, '//') !== false) {
				$this->name = str_replace('//', '/', $this->name);
			}
			
			// Load the default/home page if no real page was set
			if(empty($this->name)) {
				$this->name = $config->config()->fetch('defaults.name');
			}
			
			// Try to load the pre-configured resource object
			if($config->has_config('resource')) {
				// Load the resource
				$loader = new PluginLoader(
						$config->config()->fetch('resource'),
						$config
				);
				$resource = $loader->load('resource_page', $this, $this->name);
				
				// Check if the plugin was loaded successfully
				if($resource) {
					// Convert all language codes their standard 3-letter code
					// before using them and record their original codes
					$available = array();
					foreach($resource->get_languages() as $lang) {
						if(strpos($lang, '-') !== false) {
							list($language, $country) = explode('-', $lang, 2);
						} else {
							list($language, $country) = array($lang, null);
						}
						
						$language = is_string($language) ? strtolower($language) : null;
						$country  = is_string($country)  ? strtoupper($country)  : null;
						
						// Convert all language codes to their standard 3-letter
						// format if possible
						if(isset($language_codes_translation_table[$language])){
							$language = $language_codes_translation_table[$language];
						}
						// Convert all country codes to their standard 3-letter
						// format if possible
						if(isset($country_codes_translation_table[$country])) {
							$country = $country_codes_translation_table[$country];
						}
						
						if($country) {
							$available["{$language}-{$country}"] = $lang;
						} else {
							$available[$language] = $lang;
						}
					}
					
					// Find the first preferred language that actually exists
					foreach($this->preferred_language_list() as $lang) {
						if(isset($available[$lang])) {
							$resource->set_language($available[$lang]);
							$this->language = $lang;
							break;
						}
					}
					
					// If no language entry was found use the first available
					// if there are any languages at all
					if(!$this->language && count($available) > 0) {
						$resource->set_language(reset($available));
						$this->language = key($available);
					}
					
					// Gather length and resource information
					$this->resource = $resource;
					$this->length   = $resource->get_length();
				} else {
					// Throw an exception if the resource could not be loaded
					throw new PageStreamResourceError(
					  $config->config()->fetch('resource')
					);
				}
			} else {
				// Throw an exception if no resource is defined
				throw new PageStreamResourceError();
			}
			
			return $this;
		}
		
		
		/**
		 * Create a list of languages that the user may want to use in decending
		 * order
		 * 
		 * All languages will be returned as iso3166 language codes which might
		 * be followed by a hypen (-) and an iso639 country code. All codes
		 * will be standard 3-letter codes if it is possible. If any source used
		 * by this function uses 2-letter codes they will be converted. The
		 * special value "general" allows any language to be substituted for it.
		 * 
		 * @return array
		 */
		public function preferred_language_list() {
			// Load automatically generated region data
			static $country_codes_translation_table;
			static $language_codes_translation_table;
			if(!is_array($country_codes_translation_table)
			|| !is_array($language_codes_translation_table)) {
				require(dirname(__FILE__) . '/' . "regiondata.thin.php");
			}
			
			// Create a list of possible language identifiers
			$languages_raw = array();
			// Use URL parameters if any are given
			if(isset($this->params['language'])) {
				$languages_raw[] = $this->params['language'];
			}
			// Use HTTP Accept-Language header if it was given
			if(isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
				// Create a list of HTTP accept-language fields were the
				// importance of each entry is the key and the language it
				// represents is the value
				$accept_languages = array();
				foreach(explode(',', $_SERVER['HTTP_ACCEPT_LANGUAGE']) as $langtok) {
					if(strpos(';', $langtok) !== false) {
						list($lang, $suffix) = explode(';', $langtok, 2);
						if(substr($suffix, 0, 2) == 'q=') {
							$q = floatval(substr($suffix, 2));
						} else {
							$q = 0;
						}
					} else {
						$lang = $langtok;
						$q    = 1;
					}
					
					// Lower importance of language entry until there isn't any
					// other conflicting entry
					while(isset($accept_languages[$q])) {
						$q -= 0.01;
					}
					
					$accept_languages[$q] = $lang;
				}
				// Sort the array by the importance of each language
				ksort($accept_languages);
				
				// Append all languages found here to the array of possible
				// language entries
				$languages_raw = array_merge(
						$languages_raw, array_values($accept_languages)
				);
			}
			// Append all preferred languages
			$languages_raw = array_merge(
			  $languages_raw,
			  (array) $this->config->config()->fetch('preferred.languages')
			);
			// Append the universal "general" language
			$languages_raw[] = "general";
			
			// Convert all language entries to their 3-letter representation
			// if they aren't already and make sure there is always a country
			// unspecific entry for each language
			$languages_array = array();
			foreach($languages_raw as $lang) {
				if(strpos($lang, '-')) {
					list($language, $country) = explode('-', $lang, 2);
				} else {
					list($language, $country) = array($lang, null);
				}
				
				$language = is_string($language) ? strtolower($language) : null;
				$country  = is_string($country)  ? strtoupper($country)  : null;
				
				// Convert all language codes to their standard 3-letter format
				// if possible
				if(isset($language_codes_translation_table[$language])) {
					$language = $language_codes_translation_table[$language];
				}
				// Convert all country codes to their standard 3-letter format
				// if possible
				if(isset($country_codes_translation_table[$country])) {
					$country = $country_codes_translation_table[$country];
				}
				
				if(!isset($languages_array[$language])) {
					$languages_array[$language] = array();
				}
				if($country && !in_array($country, $languages_array[$language])) {
					$languages_array[$language][] = $country;
				}
			}
			$languages = array();
			foreach($languages_array as $language => $countries) {
				// Skip empty entries
				if(empty($language)) {
					break;
				}
				
				foreach($countries as $country) {
					$languages[] = "{$language}-{$country}";
				}
				$languages[] = $language;
			}
			
			return $languages;
		}
		
		
		/**
		 * Render and display this page's content
		 *
		 * @return bool Success?
		 */
		public function display() {
			// Determine the type of this page
			if(is_string($renderer=$this->get_header('renderer'))) {
				$renderer = strtolower($renderer);
			} else {
				$renderer = $this->config->config()->fetch('defaults.renderer');
			}
			
			// Check if the renderer defined by this page actually exists
			$loader = new PluginLoader($renderer, $this->config);
			$plugin = $loader->load('renderer', $this);
			if($plugin) {
				$plugin->display();
				return true;
			} else {
				throw new PageRendererNotFoundError($this->name, $renderer);
				return false;
			}
		}
		
		
		/**
		 * Return if the resource thinks it exists ;)
		 * 
		 * @param bool [$allow_meta_items=true]
		 *        Should a page which just provides meta-information be
		 *        displayed as non-existant?
		 * @return bool
		 */
		public function exists($allow_meta_items=true) {
			if($allow_meta_items && $this->get_header('meta-page') !== false) {
				return false;
			} else {
				return $this->resource->exists();
			}
		}
		
		
		/**
		 * Save all data that hasn't been saved yet
		 */
		public function flush() {
			// Update page headers from the header cache
			if(is_array($this->header_cache)) {
				$this->resource->set_headers($this->header_cache);
			}
			
			if(is_callable(array($this->resource, 'flush'))) {
				$this->resource->flush();
			}
		}
		
		
		/**
		 * Generate a user-friendly URL for this page
		 * 
		 * @return string|false
		 */
		public function generate_url() {
			if($this->config->config()->fetch('url.rewrite')) {
				$url = $this->config->config()->fetch('url.structure');
				$url = str_replace('%page%', $this->name, $url);
				$url = str_replace('%lang%', $this->language, $url);
				return $url;
			} else {
				return false;
			}
		}
		
		
		/**
		 * Return the real filepath of the stream
		 * 
		 * @return string|null
		 */
		public function get_filepath() {
			return $this->resource->get_real_path();
		}
		
		
		/**
		 * Return a header item of the page
		 * 
		 * @param string $name
		 *        The name of the item to retrieve
		 * @return string|false
		 */
		public function get_header($name) {
			// Check if resource has a get_header method
			if(is_callable(array($this->resource, 'get_header'))) {
				// Let resource handle this function call if it can directly
				return $this->resource->get_header($name);
			} else {
				// Load all defined headers into the cache if it hasn't been
				// done already
				if(!is_array($this->header_cache)) {
					$this->header_cache = $this->resource->get_headers();
				}
				
				// Read and return item from cache
				if(isset($this->header_cache[$name])) {
					return $this->header_cache[$name];
				} else {
					return false;
				}
			}
		}
		
		
		/**
		 * Set the value of a header item of this page
		 * 
		 * @param string $name
		 *        The name of the item to change/create
		 * @param string $value
		 *        The new value of the item
		 */
		public function set_header($name, $value) {
			// Check if resource has a set_header method
			if(is_callable(array($this->resource, 'set_header'))) {
				// Let resource handle this function call if it can directly
				$this->resource->set_header($name, $value);
			} else {
				// Write item into cache
				$this->header_cache[$name] = $value;
			}
		}
		
		
		/**
		 * Return a string which uniquly identifies the resource loaded
		 * 
		 * @return string
		 */
		public function get_identifier() {
			return "{$this->language}:{$this->name}";
		}
		
		
		/**
		 * Return the resource object actually used
		 * 
		 * @return resource|null
		 */
		public function get_resource() {
			if(is_callable(array($this->resource, 'get_resource'))) {
				return $this->resource->get_resource();
			} else {
				return null;
			}
		}
		
		
		/**
		 * Read $length characters starting at $start
		 * 
		 * If $length is null all remaining characters will be read.
		 * 
		 * @param integer [$start=0]
		 *        Where to start from reading
		 * @param integer [$length=ꝏ]
		 *        How many characters to read
		 * @return string|null
		 */
		public function read($start=0, $length=null) {
			$data = $this->resource->read($start, $length);
			
			//TODO: Implement output filter plugins
			//foreach(PluginLoader::list_plugins())
			
			return $data;
		}
		
		
		/**
		 * Render this page but do not emit any headers or content
		 * 
		 * @return string
		 */
		public function render() {
			// Enable output buffering
			ob_start();
			
			// "Display" the page into the output buffer
			$this->display();
			
			// Return buffer contents
			return ob_get_clean();
		}
		
		
		/**
		 * Collect and return the page's stat information
		 *
		 * @see http://php.net/manual/function.stat.php
		 * @return array
		 */
		public function stat() {
			// Check if the resource has its own stat function
			if(is_callable(array($this->resource, 'stat'))) {
				// Make sure the stat is completely filled and return it
				return parent::stat_generator($this->resource->stat());
			} elseif(file_exists($this->resource->get_real_path())) {
				// Get stat information from the file actually used
				$stat = stat($this->resource->get_real_path());
				$stat['size'] = $this->length;
				return $stat;
			} else {
				// Just fill in the bits we know and let stat_generator
				// do the rest
				$stat = array(
				  'size' => $this->length
				);
				return parent::stat_generator($stat);
			}
		}
		
		
		/**
		 * Rename the page of the open page object
		 * 
		 * @param string $name      The new name of the page
		 * @param bool   $overwrite Should be considered to overwrite an other page?
		 * @return bool Success?
		 */
		public function rename($name, $overwrite=false) {
			if($this->name == $name) {
				return true;
			}
			
			// Generate new filepath
			$filepath = dirname($this->get_path()) . '/';
			
			// Keep extension if any
			$name_parts = explode('.', basename($this->get_path()));
			if(count($name_parts) > 1) {
				$filepath .= $name . '.' . end($name_parts);
			} else {
				$filepath .= $name;
			}
			
			// Remove original file if overwriting is permitted
			if(file_exists($filepath)) {
				if($overwrite && (is_writeable($filepath) || @chmod($filepath, 0666))) {
					unlink($filepath);
				} else {
					return false;
				}
			}
			
			// Do real move
			if(is_writeable(dirname($filepath)) || @chmod(dirname($filepath), 0666)) {
				rename($this->get_path(), $filepath);
			} else {
				return false;
			}
			
			// Update internal variables
			$this->name = $name;
			$this->path = $filepath;
			
			return true;
		}
		
		
		/**
		 * Change the content of the page starting from position $start
		 * 
		 * @param integer [$start=0]
		 *        Where to start from writing
		 * @param string  $content
		 *        What to write
		 * @return bool Success?
		 */
		public function write($start=0, $content) {
			$this->buffering = true;
			
			if($start > 0) {
				$content = $this->read(0, $start) . $content . $this->read($start+strlen($content));
			}
			
			return $this->resource->write($content);
		}
		
		
		
		
		
		/**
		 * Return the language of this page
		 * 
		 * @tags api
		 * @return string
		 */
		public function get_language() {
			return (string) $this->language;
		}
		
		
		/**
		 * Return the name of this page
		 * 
		 * @tags api
		 * @return string
		 */
		public function get_name() {
			return (string) $this->name;
		}
		
		
		/**
		 * Return the array used to initialize this page
		 * 
		 * @tags api
		 * @return array
		 */
		public function get_params() {
			return (array) $this->params;
		}
		
		
		/**
		 * Try to find title for the page
		 * 
		 * @param string $page_name
		 *        The name of the page which should be examined
		 * @param bool   $title_preferred
		 *        Should title field be preferred over the heading?
		 * 
		 * @tags api
		 * @return string|false
		 */
		public function find_title($page_name=null, $title_preferred=false) {
			if(is_string($page_name) && $page_name != $this->name) {
				$page = new self();
				$page = $page->initialize(
				  $this->config,
				  null,
				  $page_name,
				  'r',
				  array('language' => $this->language)
				);
			} else {
				$page =&$this;
			}
				
			
			// Configuration with title preferred
			if($title_preferred) {
				if($page->get_header('title')) {
					return $page->get_header('title');
				} elseif($page->get_header('header')) {
					return $page->get_header('header');
				} else {
					return false;
				}
			// Configuration with header preferred
			} else {
				if($page->get_header('header')) {
					return $page->get_header('header');
				} elseif($page->get_header('title')) {
					return $page->get_header('title');
				} else {
					return false;
				}
			}
		}
		
		
		/**
		 * Generate an title for the current page
		 *
		 * This method will return a string if $sep contains a valid string or
		 * an array with all parts of the page title otherwise.
		 * 
		 * If $is_title is true then the page's title field will be used instead
		 * of its heading field.
		 *
		 * Example of the array parts ($vendor=true, $reverse=false):
		 * array('MyCompany', 'Homepage', 'News', 'The big foobar came!')
		 *
		 * @param string|null $sep
		 *        The separator of the hierarchy parts
		 * @param bool        $is_title
		 *        Should the page title instead of the page heading be used?
		 * @param bool        $reverse
		 *        Should the the list be reversed before returning?
		 * @param bool|string $vendor
		 *        Add vendor name to the array?
		 *        (Either the default one or one given by a string)
		 * @tags api
		 * @return array|string
		 */
		public function get_page_title($sep=null, $is_title=false, $reverse=false, $vendor=false) {
			// A list of title parts
			$parts = array();
			
			// Check if the vendor is given
			if(is_string($vendor)) {
				$parts[] = $vendor;
			// Check if the default vendor is requested
			} elseif($vendor) {
				$parts[] = $this->config->config()->fetch('name');
			}
			
			// Iterate through the parts of the page name and generate a title
			// string for each of them
			$name = '';
			foreach(explode('/', $this->name) as $subname) {
				// Add the new part of the page name to the last name
				//  ($name="abc" & $subname="def" makes $name="abc/def")
				$name = empty($name) ? $subname : "{$name}/{$subname}";
				// Retrieve the title of the page
				$title = $this->find_title($name, $is_title);
				// Add the title to the array of title parts if it exists
				$parts[$name] = $title ? $title : $name;
			}
			
			// Reverse the array if requested
			$parts = $reverse ? array_reverse($parts) : $parts;
			
			if(is_string($sep)) {
				return implode($parts, $sep);
			} else {
				return $parts;
			}
		}
	}
