<?php
	/**
	 * Default plugin handling page streams
	 * 
	 * @package aCMS-plugins
	 */
	/**
	 * The basic layout for page handler plugins
	 * 
	 * Page handler plugins are plugins which take the raw data of a page and
	 * transform it into a complete rendered page.
	 * 
	 * @param Configuration $config The current configuration object
	 * @param Page          $page   The object loading this plugin
	 */
	interface PluginLayoutRenderer extends PluginLayout {
		public function __construct($config, $page);
		
		/**
		 * Return the generated content of the page
		 * 
		 * @return string
		 */
		public function get_content();
		
		/**
		 * Display the content of the page
		 */
		public function display();
	}
	
	
	/**
	 * The plugin layout for output plugins
	 * 
	 * The main purpose of an output plugin is to generate content for renderer
	 * plugins. Instead of having different plugins for each rendering engine
	 * output plugins provide an unified API that has bindings for all engines
	 * available.
	 * 
	 * The result produced by {@link process} must be a string containing only
	 * plain text without any kind of markup (NO HTML!).
	 */
	interface PluginLayoutOutput extends PluginLayoutPlugin {
		/**
		 * Can the output plugin be cached?
		 * 
		 * @return bool
		 */
		public function cacheable($content, $data);
		
		
		/**
		 * Process the given data and return a result
		 * 
		 * @return string
		 */
		public function process($content, $data);
	}
	
	/**
	 * The plugin layout for HTML output plugins
	 * 
	 * The result produced by {@link process} must be a string containing only
	 * valid HTML.
	 */
	interface PluginLayoutOutputHtml extends PluginLayoutOutput {}
	
	
	/**
	 * The layout for filters
	 */
	interface PluginLayoutFilter extends PluginLayoutPlugin {
		/**
		 * Return the filtered content
		 * 
		 * @return string
		 */
		public function process($content);
	}
	
	/**
	 * The layout for content filters
	 */
	interface PluginLayoutFilterOutput extends PluginLayoutFilter {}
	
	
	/**
	 * The layout for the resources providing page extensions
	 */
	interface PluginLayoutResourcePage extends PluginLayoutResource {
		/**
		 * Return an array of headers which were defined within the page
		 * 
		 * @return array
		 */
		public function get_headers();
		
		/**
		 * Return an array of all languages supported by the current page
		 * 
		 * Each language must be a valid ISO-3166-1 2-letter or 3-letter
		 * language code. Each language may be followed by a hypen (-) or
		 * underscore (_) separating it from a valid ISO-639-2 2-letter or
		 * 3-letter country code.
		 * 
		 * Note: Internally all codes are converted to 3-letter codes were
		 * possible, so de-AT and deu-AUT are treated in the same way.
		 * 
		 * @return array
		 */
		public function get_languages();
		
		/**
		 * Return the language used when reading this page
		 * 
		 * @return string
		 */
		public function get_language();
		
		/**
		 * Set the language to use when reading this page
		 * 
		 * @param string $language
		 *        The name of the language to use
		 */
		public function set_language($language);
		
		/**
		 * Return the value of a specific page header
		 * 
		 * Only implement this function if your resource supports reading
		 * a single header by itself without having to read the whole headers
		 * array.
		 * 
		 * @param string $name
		 *        The name of the header item
		 * @return string|false
		 */
		//public function get_header($name);
		
		
		/**
		 * Change the value of all page headers
		 * 
		 * @param array $headers
		 *        An array of headers
		 */
		public function set_headers(array $headers);
		
		
		/**
		 * Change the value of a single page header
		 * 
		 * Only implement this function if your resource supports writing
		 * a single header by itself without having to write all other headers.
		 * 
		 * @param string $name
		 *        The name of the header item
		 * @param string $value
		 *        The new value of the header item
		 */
		//public function set_header($name, $value);
	}
	
	
	
	/**
	 * API function for output plugins retriving an item from the $params
	 * array
	 * 
	 * First this function tries to find the keyword (string) offset of this
	 * item then the numeric (integer) offset of this item if none of the
	 * could be found the default value will be returned.
	 * 
	 * @param array   $params
	 *        The parameters to read
	 * @param int     $index
	 *        The possible numeric offset (e.g.: 3)
	 * @param string  $name
	 *        The possible keyword offset (e.g.: "kwarg")
	 * @param string [$default=null]
	 *        An alternate value to use if none of the values above match
	 */
	function find_param(array $params, $index, $name, $default=null) {
	 	// Try to find the item by the name given
	 	if(isset($params[$name])) {
	 		return $params[$name];
	 	}
		
		// Try to find the item by the index given
		if(isset($params[$index])) {
	 		return $params[$index];
	 	}
		
		return $default;
	}
	
	
	/**
	 * API function for output plugins retriving multiple items from the
	 * $params array
	 * 
	 * The $params array is a key/value array which has the possible item
	 * names as keys and the default values as values. The offset of the
	 * item within the array is used as the numeric item value.
	 * 
	 * @param array $params
	 *        An array of parameters to use as input
	 * @param array $definitions
	 *        An array of item definitions explained above
	 * @return array
	 */
	function find_params_assoc(array $params, array $definitions) {
		$assoc = array();
		
		$x = 0;
		foreach($definitions as $name => $default) {
			// Check for a keyword argument
			if(isset($params[$name])) {
				// Set item data from the keyword element
				$assoc[$name] = $params[$name];
			// Check for a numeric argument
			} elseif(isset($data[$x])) {
				$assoc[$name] = $params[$x];
			} else {
				$assoc[$name] = $default;
			}
			
			$x++;
		}
		
		return $assoc;
	}
	
	
	/**
	 * @alias find_params_assoc
	 * 
	 * This function does the same as find_params_assoc except that it does not
	 * return any names in the resulting array. Useful for list();
	 */
	function find_params(array $data, array $definitions) {
		return array_values(find_params_assoc($data, $definitions));
	}
	
	
	
	/**
	 * Main class of the internal page plugin
	 */
	class PageMain implements PluginLayoutMain {
		/**
		 * @param Configuration $config The current configuration object
		 * @param PluginLoader  $loader The loading PluginLoader instance
		 */
		public function __construct($config, $loader) {}
		
		/**
		 * @return array
		 */
		public function supported() {
			return array('stream');
		}
		
		/**
		 * Return the current version of this plugin
		 * 
		 * @return string
		 */
		public function get_version() {
			return '1';
		}
		
		/**
		 * Is it always required to include the main plugin file?
		 * 
		 * @return bool
		 */
		public function include_main() {
			return true;
		}
	}
