<?php
	/**
	 * 
	 *
	 * @package aCMS
	 */
	
	/**
	 * This plugin class allows you to parse and use INI/configuration files as
	 * source of your configuration data
	 */
	class IniConfigurationPlugin implements PluginLayoutConfiguration {
		/**
		 * The location of the configuration file
		 *
		 * @var string
		 */
		protected $filepath = '';
		
		/**
		 * The data of the configuration file
		 *
		 * @var array
		 */
		protected $sections = array();
		
		
		
		/**
		 * @param Configuration           $config     The current configuration
		 * @param mixed                   $loader     The loading object
		 * @param ConfigurationConnection $connection The data class containing
		 * the information on how to connect
		 */
		public function __construct($config, $loader, $connection) {
			$this->loader   = $loader;
			$this->filepath = $connection->filepath;
			
			// Load the configuration file contents
			$content = file_get_contents($this->filepath);
			
			// Add dummy header to the end of the file to make sure the last
			// section gets matched/parsed too
			$content .= "\n[IGNORE]\n";
			
			// Generate multi-line pattern
			$pattern  ='/^(?:[;#]+[ \t[:punct:]]+' . "\n";
			$pattern .= '[;#]+[ \t[:punct:]]+(.+?):?[ \t[:punct:]]+' . "\n";
			$pattern .= '[;#]+[ \t[:punct:]]+'     . "\n)?";
			$pattern .= '[ \t]*\[(\w+)\][ \t]*'    . "\n";         
			$pattern .= '(.*?)'                    . "\n";
			$pattern .= '[ \t]*\[\w+\][ \t]*' . '/is';
			
			// Match content with pattern
			while(preg_match($pattern, $content, $match, PREG_OFFSET_CAPTURE)) {
				// Create new configuration section
				$section = new ConfigurationSection($match[2][0], $match[1][0]);
				// Store multi-line descriptions
				$desc    = '';
				
				// Parse configuration file line by line
				foreach(explode("\n", $match[3][0]) as $line) {
					// What the different $result parts mean:
					//$name   = $result[2];
					//$value  = self::parse_field($result[3]):
					//$active = empty($result[1]);
					
					// Trim line to make matches more accurate
					$line = trim($line);
					
					// Skip empty lines
					if(empty($line)) {
						continue;
					}
					
					// Try to parse this line as key-value-pair
					$regex = '/([;#]*)[ \t]*([\w.]*)\s*=\s*("[^"]*"|[^\n]*)/is';
					if(preg_match($regex, $line, $result)) {
						// Try to to find duplicates
						foreach($section->get_items() as $index => $item) {
							if($item->get_name() == $result[2]) {
								// Ignore current item if the version which
								// occurs earlier is active while the current
								// version is inactive
								if($item->is_active() && !empty($result[1])) {
									continue 2;
								}
							}
						}
						
						// Append new item to item list
						$section->append_item(new ConfigurationItem(
						   $result[2],                    // Key
						   substr($desc, 0, -1),          // Description
						   self::parse_field($result[3]), // Value
						   empty($result[1])));           // Active
						
						// Clear description for next item
						$desc = '';
					// Try to parse this line as comment
					} elseif(in_array(substr($line, 0, 1), array(';', '#'))) {
						// Remove a suffixing colon
						if(substr($line, -1) == ':') {
							$desc .= ltrim(substr($line, 1, -1)) . "\n";
						} else {
							$desc .= ltrim(substr($line, 1)) . "\n";
						}
					// Invalid contents
					} else {
						$this->sections = false;
						return;
					}
				}
				
				// Cut off the current section from the content
				$content = ltrim(substr($content,
				                        $match[3][1] + strlen($match[3][0])));
				
				// Add the current section to the data
				$this->sections[] = $section;
			}
		}
		
		
		/**
		 * Parse a string field value and give back its actual contents
		 * 
		 * @param string $field A string containing the field to parse
		 * @return mixed
		 */
		public static function parse_field($field) {
			$field = trim($field);
			
			// Number
			if(is_numeric($field)) {
				return floatval($field);
			// List
			} elseif(substr($field, 0, 1) == '[' && substr($field, -1) == ']') {
				$field = substr($field, 1, -1);
				$list  = array();
				
				preg_match_all('/([\'"].+[\'"]|.+)\s*,\s*/iU',"$field,",$items);
				foreach($items[1] as $item) {
					$list[] = self::parse_field($item);
				}
				
				return $list;
			} else {
			// Boolean
				switch(strtolower($field)) {
					// True
					case 'true':
					case 'yes':
						return true;
					break;
					
					// False
					case 'false':
					case 'no':
						return false;
					break;
				}
			// String
				// Quoted
				if((substr($field, 0, 1) == '"' && substr($field, -1) == '"')
				|| (substr($field, 0, 1) == "'" && substr($field, -1) == "'")) {
					return substr($field, 1, -1);
				// Unquoted
				} else {
					return $field;
				}
			}
		}
		
		
		/**
		 * Return a list of configuration sections
		 * 
		 * @uses ConfigurationSection
		 * @uses ConfigurationItem
		 * 
		 * @return array
		 */
		public function get_sections() {
			return $this->sections;
		}
		
		
		/**
		 * Return the last modification time of the file
		 * 
		 * @return integer
		 */
		public function get_timestamp() {
			return filemtime($this->filepath);
		}
	}
