<?php
	require_component("AOF/AOFError.class.php");
	
	
	/**
	 * File resource plugin for page streams
	 */
	class FilePageResourcePlugin implements PluginLayoutResourcePage {
		/**
		 * A list of possible places to look for pages
		 * 
		 * @var array
		 */
		public static $path = array();
		
		
		/**
		 * The content of the page
		 * 
		 * @var string
		 */
		protected $content;
		
		/**
		 * The headers array of this page
		 * 
		 * @var array
		 */
		protected $headers;
		
		/**
		 * The name of the language to use
		 * 
		 * @var string
		 */
		protected $language = '';
		
		/**
		 * A cached version of all available language paths
		 */
		private $language_paths;
		
		/**
		 * The name of the page represented by this resource
		 * 
		 * @var string
		 */
		protected $name = '';
		
		
		
		/**
		 * Parse the headers of a page and give back a list containing an array
		 * of header fields and the remaining body part of the page file
		 * 
		 * @param string $content The content of the page file to parse
		 * 
		 * @return array(array,string)
		 */
		public static function page_parse($content) {
			$headers = array();
			$lines   = explode("\n", (string) $content);
		
			foreach($lines as $line) {
				$line = trim($line);
				
				// Check for the information comments
				if(strpos($line, '<!--$') === 0 && substr($line, -4) == '$-->'){
					// Cut off the comment parts
					$line = substr($line, 5);
					$line = substr($line, 0, -4);
					
					$parts = explode(':', $line, 2);
					$name  = trim(strtolower($parts[0]));
					$value = isset($parts[1]) ? trim($parts[1]) : true;
					
					// Assign the name and value to the header list
					$headers[$name] = $value;
					
					// Remove the line from the line list
					$lines = array_slice($lines, 1);
				} elseif(empty($line)) {
					// Also remove empty header lines
					$lines = array_slice($lines, 1);
				} else {
					// Stop at the first line of content
					break;
				}
			}
			
			return array($headers, implode($lines, "\n"));
		}
		
		
		/**
		 * Generate a page file
		 * 
		 * @param array  $headers An array of headers (array($name => $value))
		 * @param string $content The content of the page to generate
		 * 
		 * @return string The whole page body
		 */
		public static function page_generate($headers, $content) {
			$body = '';
			
			foreach((array) $headers as $name => $value) {
				if($value !== NULL) {
					$body .= "<!--\$ $name: $value \$-->\n";
				} else {
					$body .= "<!--\$ $name \$-->\n";
				}
			}
			
			if($headers) {
				$body .= "\n\n";
			}
			
			$body .= (string) $content;
			
			return $body;
		}
		
		
		
		/**
		 * Parse the page file currently opened and fill the $this->content and
		 * $this->headers variables
		 * 
		 * @internal
		 * @return bool Success?
		 */
		protected function pagefile_parse() {
			if($this->get_real_path()) {
				// Read and parse page
				$page = self::page_parse(
				  file_get_contents($this->get_real_path())
				);
				
				// Check if parsing was successful
				if(is_array($page) && count($page) === 2) {
					// Fill the $this->content and $this->headers variables
					list($this->headers, $this->content) = $page;
					
					return true;
				} else {
					return false;
				}
			} else {
				return false;
			}
		}
		
		
		/**
		 * Overwrite/create the page file on the harddisk with the information
		 * given in the $this->content and $this->headers variables
		 * 
		 * @internal
		 * @return bool Success?
		 */
		protected function pagefile_generate() {
			// Check if we may write to the real file
			if(!is_writeable($this->get_real_path())
			&& !@chmod($this->get_real_path(), 0666)) {
				return false;
			}
			
			// Generate and write page to file
			return (bool) file_put_contents(
			  $this->get_real_path(),
			  self::page_generate($this->headers, $this->content)
			);
		}
		
		
		
		
		
		
		/**
		 * @param Configuration     $config
		 *        The current configuration object
		 * @param PageStreamPlugin  $loader
		 *        The object loading this plugin
		 * @param string            $name
		 *        The name of the page to load
		 */
		public function __construct($config, $loader, $name) {
			$config = $config->create_new('page.');
			
			// Add all configuration path directories to the page file
			// search path
			foreach((array) $config->get_config('directories') as $directory) {
				if(!in_array($directory, self::$path, true)) {
					self::$path[] = $directory;
				}
			}
			
			// Waves (~) are used internally in filenames instead of slashes (/)
			// as slashes indicate directories and are therefore not
			// allowed within a filename
			$name = str_replace('/', '~', $name);
			
			$this->loader = $loader;
			$this->name   = $name;
		}
		
		
		/**
		 * Return if the requested resource actually exists
		 * 
		 * @return bool
		 */
		public function exists() {
			return (bool) $this->get_real_path();
		}
		
		
		/**
		 * Return an array of headers which were defined within the page
		 * 
		 * @return array
		 */
		public function get_headers() {
			// Parse file first if it wasn't parsed already
			if(!is_array($this->headers) || !is_string($this->content)) {
				if($this->pagefile_parse()) {
					return false;
				}
			}
			
			return $this->headers;
		}
		
		
		/**
		 * Change the value of all page headers
		 * 
		 * @param array $headers
		 *        An array of headers
		 */
		public function set_headers(array $headers) {
			$this->headers = $headers;
			
			return $this->pagefile_generate();
		}
		
		
		/**
		 * Return an array of all languages supported by this page where the
		 * array keys are the language names and the array values are the paths
		 * of the files holding them
		 * 
		 * array("$language" => "$filepath", ...)
		 * 
		 * @return array
		 */
		public function get_language_paths() {
			if(is_array($this->language_paths)) {
				return $this->language_paths;
			}
			
			// Create a list of available languages if the page directory
			// is defined
			$language_paths = array();
			foreach((array) self::$path as $directory) {
				// Skip all path entries that do not define a directory for the
				// current page name
				if(!is_dir("{$directory}/{$this->name}")) {
					continue;
				}
				
				foreach(scandir("{$directory}/{$this->name}") as $filename) {
					// Skip hidden files and directories
					if(substr($filename, 0, 1) == '.'
					|| !is_file("{$directory}/{$this->name}/{$filename}")) {
						continue;
					}
					
					// Remove the file extension
					if(strpos($filename, '.') !== null) {
						$language = explode('.', $filename);
						$language = array_slice($language, 0, -1);
						$language = implode($language, '.');
					} else {
						$language = $filename;
					}
					
					// Add language to language list
					if(!isset($language_paths[$language])) {
						$language_paths[$language] = 
						  "{$directory}/{$this->name}/{$filename}";
					}
				}
			}
			
			$this->language_paths = $language_paths;
			return $this->language_paths;
		}
		
		
		/**
		 * Return an array of all languages supported by this page
		 * 
		 * @return array
		 */
		public function get_languages() {
			return array_keys($this->get_language_paths());
		}
		
		
		/**
		 * Return the language used when reading this page
		 * 
		 * @return string
		 */
		public function get_language() {
			return $this->language;
		}
		
		
		/**
		 * Set the language to use when reading this page
		 * 
		 * @param string $language
		 *        The name of the language to use
		 */
		public function set_language($language) {
			$this->language = $language;
		}
		
		
		/**
		 * Return the length of the content of the current resource
		 * 
		 * @return integer
		 */
		public function get_length() {
			// Parse file first if it wasn't parsed already
			if(!is_array($this->headers) || !is_string($this->content)) {
				if($this->pagefile_parse()) {
					return false;
				}
			}
			
			return strlen($this->content);
		}
		
		
		/**
		 * Return the real filepath used to access the resource used here
		 * 
		 * @return string|false
		 */
		public function get_real_path() {
			// We cann't generate a filepath if we don't know the language yet
			if(!$this->language) {
				return false;
			}
			
			$language_paths = $this->get_language_paths();
			if(isset($language_paths[$this->language])) {
				return $language_paths[$this->language];
			} else {
				return false;
			}
		}
		
		
		/**
		 * Read $length characters starting at $start
		 * 
		 * If $length is null all remaining characters will be read.
		 * 
		 * @param integer [$start=0]
		 *        Where to start from reading
		 * @param integer [$length=]
		 *        How many characters to read
		 * @return string|false
		 */
		public function read($start=0, $length=null) {
			// Parse file first if it wasn't parsed already
			if(!is_array($this->headers) || !is_string($this->content)) {
				if($this->pagefile_parse()) {
					return false;
				}
			}
			
			return substr($this->content, $start, $length);
		}
		
		
		/**
		 * Change the content of the resource starting at position $start
		 * 
		 * @param integer [$start=0]
		 *        Where to start from writing
		 * @param string  $content
		 *        What to write
		 * @return bool Success?
		 */
		public function write($start=0, $content) {
			// Update cached content
			$this->content = $this->read(0, $start) . $content . $this->read($start+strlen($content));
			
			return $this->pagefile_generate();
		}
	}
