<?php
	require_component("AOF/AOFError.class.php");
	
	/**
	 * File resource plugin for page streams
	 */
	class FileContentResourcePlugin implements PluginLayoutResourceContent {
		/**
		 * A list of directories which should be searched for resources
		 */
		static public $path = array();
		
		
		/**
		 * The path at which the client can access the resource
		 * 
		 * @var string
		 */
		protected $clientpath = null;
		
		/**
		 * The full path of the resource which is accessed
		 * 
		 * @var string
		 */
		protected $fullpath = null;
		
		/**
		 * The name of the content file resource
		 * 
		 * @var string
		 */
		protected $name = '';
		
		
		
		/**
		 * Add a directory to the search path
		 * 
		 * @param string $directory
		 *        The new directory path to add
		 */
		static function add_path_directory($directory) {
			$directory = fixpath($directory);
			
			if(!isset(self::$path[$directory])) {
				$client_prefix = null;
				if(strpos($directory, '://', 1) > -1) {
					// Use the absolute URL given as location were the client
					// can help himself
					$client_prefix = $directory;
				} else {
					// Try to find the directory given on the current server
					// and use propagate it to the client if it is on the same
					// server
					$srvparts = explode('/', fixpath($_SERVER['DOCUMENT_ROOT'], '/'));
					$dirparts = explode('/', str_replace("\\", '/', $directory));
					$matches  = true;
					for($x=0, $max=count($srvparts); $x < $max; $x++) {
						if($srvparts[$x] != $dirparts[$x]) {
							$matches = false;
							break;
						}
					}
					
					if($matches) {
						$client_prefix = '/' . implode(
							array_slice($dirparts, count($srvparts)), '/'
						);
					}
				}
				
				self::$path[$directory] = array(
					'client_prefix' => $client_prefix
				);
			}
		}
		
		
		/**
		 * @param Configuration       $config
		 *        The current configuration object
		 * @param ContentStreamPlugin $loader
		 *        The object loading this plugin
		 * @param string              $name
		 *        The name of the content resource to load
		 */
		public function __construct($config, $loader, $name) {
			$config = $config->create_new('content.');
			
			// Add all configuration path directories to the content file
			// search path
			foreach((array) $config->get_config('directories') as $directory) {
				self::add_path_directory($directory);
			}
			
			$this->name = $name;
			
			// Find the content file requested and make sure that the final path
			// is inside of one of the directories given
			foreach(self::$path as $basepath => $data) {
				$basepath = realpath($basepath);
				// Skip non-existing and non-directory filepaths
				if(is_null($basepath) || !is_dir($basepath)) {
					continue;
				}
				
				// Skip file content directories which don't contain the
				// requested file
				$fullpath = realpath("{$basepath}/{$this->name}");
				if(is_null($fullpath) || !is_file($fullpath)) {
					continue;
				}
				
				// Make sure the full filepath is actually located within the
				// path directory
				if(substr($fullpath, 0, strlen($basepath)) != $basepath) {
					continue;
				}
				
				$this->fullpath = $fullpath;
				if($data['client_prefix']) {
					$this->clientpath="{$data['client_prefix']}/{$this->name}";
				}
				
				break;
			}
		}
		
		
		/**
		 * Return if the requested resource actually exists
		 * 
		 * @return bool
		 */
		public function exists() {
			return (bool) $this->get_real_path();
		}
		
		
		/**
		 * Return the file content URL which could be opened instead
		 * 
		 * @return string|null
		 */
		public function generate_url() {
			return $this->clientpath;
		}
		
		
		/**
		 * Return the length of the content of the current resource
		 * 
		 * @return integer
		 */
		public function get_length() {
			if($this->exists()) {
				return filesize($this->get_real_path());
			} else {
				return null;
			}
		}
		
		
		/**
		 * Return the MIME type of this resource
		 * 
		 * @return string
		 */
		public function get_mime_type() {
			if(function_exists('mime_content_type')) {
				return mime_content_type($this->get_real_path());
			} else {
				return finfo::file($this->get_real_path(), FILEINFO_MIME_TYPE);
			}
		}
		
		
		/**
		 * Return the real filepath used to access the resource used here
		 * 
		 * @return string|null
		 */
		public function get_real_path() {
			return $this->fullpath;
		}
		
		
		/**
		 * Read $length characters starting at $start
		 * 
		 * If $length is null all remaining characters will be read.
		 * 
		 * @param integer [$start=0]
		 *        Where to start from reading
		 * @param integer [$length=]
		 *        How many characters to read
		 * @return string|false
		 */
		public function read($start=0, $length=null) {
			if(!is_null($length) && $length < 1) {
				$data = '';
			} else {
				$handle = fopen($this->get_real_path(), 'r');
				fseek($handle, $start, SEEK_SET);
				if(is_null($length)) {
					$data = fread($handle, filesize($this->get_real_path()));
				} else {
					$data = fread($handle, $length);
				}
				fclose($handle);
			}
			
			return $data;
		}
		
		
		/**
		 * Return the stat information related to the file requested
		 * 
		 * @return stat
		 */
		public function stat() {
			return stat($this->get_real_path());
		}
		
		
		/**
		 * Change the content of the resource starting at position $start
		 * 
		 * @param integer [$start=0]
		 *        Where to start from writing
		 * @param string  $content
		 *        What to write
		 * @return bool Success?
		 */
		public function write($start=0, $content) {
			return file_put_contents(
			  $this->get_real_path(),
			  $this->read(0, $start) . $content . $this->read($start+strlen($content))
			);
		}
	}
