<?php
	class ContentStreamPlugin extends PluginHelperStream implements PluginLayoutOutputStream {
		/**
		 * The content resource which is used
		 * 
		 * @var mixed
		 */
		protected $resource;
		
		
		
		/**
		 * @param Configuration $config 
		 * @param mixed         $loader
		 * @param string        $path
		 * @param string        $mode
		 * @param array         $params
		 */
		public function initialize($config, $loader, $path, $mode='r+', array $params=array()) {
			$this->config = $config;
			$this->loader = $loader;
			
			$url = parse_url($path);
			// Combine host and path
			if(isset($url['host']) && isset($url['path'])) {
				$this->location = trim("{$url['host']}/{$url['path']}");
			} elseif(isset($url['path'])) {
				$this->location = trim($url['path']);
			} elseif(isset($url['host'])) {
				$this->location = trim($url['host']);
			} else {
				$this->location = '';
			}
			
			$this->location = str_replace("\\", '/', $this->location);
			
			// Cut off all slashes at the beginning of the location
			while(substr($this->location, 0, 1) == '/') {
				$this->location = substr($this->location, 1);
			}
			// Cut off all slashes at the end of the location
			while(substr($this->location, -1) == '/') {
				$this->location = substr($this->location, 0, strlen($this->location)-1);
			}
			// Strip all double slashes
			while(strpos($this->location, '//') !== false) {
				$this->location = str_replace('//', '/', $this->location);
			}
			
			// Load all resource objects
			foreach($this->config->config()->fetch('resources') as $name) {
				// Load the resource
				$resource = PluginLoader::load_plugin(
				  $name,
				  'resource_content',
				  $this->config,
				  $this,
				  $this->location
				);
				
				if(is_object($resource) && $resource->exists()) {
					$this->resource = $resource;
					break;
				}
			}
		}
		
		
		/**
		 * Display the content data
		 */
		public function display() {
			header('Content-Type', $this->resource->get_mime_type());
			
			print($this->read());
		}
		
		
		/**
		 * Is this content stream associated with an existing resource?
		 * 
		 * @return bool
		 */
		public function exists() {
			return is_object($this->resource) && $this->resource->exists();
		}
		
		
		/**
		 * Tell the stream to flush any unsaved chunks of data
		 */
		public function flush() {
			if(function_exists(array($this->resource, 'flush'))) {
				$this->resource->flush();
			}
		}
		
		
		/**
		 * Generate an alternate file URL, if possible
		 * 
		 * @return string|null
		 */
		public function generate_url() {
			if(method_exists($this->resource, 'generate_url')) {
				return $this->resource->generate_url();
			} else {
				return null;
			}
		}
		
		
		/**
		 * Return the filepath of the file actually opened, if any
		 * 
		 * @return string|null
		 */
		public function get_filepath() {
			if(method_exists($this->resource, 'get_real_path')) {
				return $this->resource->get_real_path();
			} else {
				return null;
			}
		}
		
		
		/**
		 * Return a string which uniquly identifies the resource loaded
		 * 
		 * @return string
		 */
		public function get_identifier() {
			return urlencode($this->location);
		}
		
		
		/**
		 * Return the PHP resource actually opened, if any
		 * 
		 * @return resource|null
		 */
		public function get_resource() {
			if(method_exists($this->resource, 'get_resource')) {
				return $this->resource->get_resource();
			} else {
				return null;
			}
		}
		
		
		/**
		 * Read data some or all data from the stream
		 * 
		 * @param integer [$start=0]
		 *        The character from which to start from reading
		 * @param integer [$length=]
		 *        How many bytes to read
		 */
		public function read($start=0, $length=null) {
			return $this->resource->read($start, $length);
		}
		
		
		/**
		 * Return the whole content data
		 * 
		 * @return string
		 */
		public function render() {
			return $this->read();
		}
		
		
		/**
		 * Generate the stat data associated with this content element
		 * 
		 * @see http://php.net/manual/function.stat.php
		 */
		public function stat() {
			if($this->resource) {
				if(method_exists($this->resource, 'stat')) {
					// Use provided stat information, if any
					$stat = (array) $this->resource->stat();
				} elseif($resource=$this->get_resource()) {
					// Use information provided through resource, if possible
					$stat = fstat($resource);
				} elseif($stat=stat($this->get_real_path())) {
					// Use information provided through real filepath, if it is,
					// and it doesn't have to be, valid
				} else {
					// Just use all information provided to generate at least
					// a little bit of stat data
					$stat = array('size' => $this->resource->get_length());
				}
			} else {
				// Generate "empty" stat data if the is no resource which could
				// provide any
				$stat = array();
			}
			
			return PluginHelperStream::stat_generator($stat);
		}
		
		
		/**
		 * Write some data to the stream
		 * 
		 * @param integer [$start=0]
		 *        Where to start writing from
		 * @param string   $data
		 *        The data to write
		 */
		public function write($start=0, $data) {
			return $this->resource->write($start, $data);
		}
	}
