<?php
	/**
	 * The plugin loader API
	 *
	 * @package aCMS
	 * @subpackage aCMS-base
	 */
	
	require_once(dirname(__FILE__) . '/' . "../AOF/AOFError.class.php");
	require_once(dirname(__FILE__) . '/' . "../report.inc.php");
	
	
	
	
	
	/**
	 * Exception base class for all plugin related errors
	 */
	class PluginError extends AOFError {
		/**
		 * The title of the error page
		 * 
		 * @var string
		 */
		protected $title = 'Error loading plugin';
		
		
		
		/**
		 * The name of the plugin which caused this exception
		 * 
		 * @var string
		 */
		public $name = '';
		
		
		
		/**
		 * @param string $name
		 *        The name of the plugin which caused this exception
		 */
		public function __construct($name, $message) {
			$this->name = $name;
			
			parent::__construct("Error in plugin '{$name}': {$message}");
		}
	}
	
	/**
	 * Exception class called whenever a plugin is loaded which doesn't exist
	 */
	class PluginNotFoundError extends PluginError {
		/**
		 * @param string $name
		 *        The name of the plugin which couldn't be found
		 */
		public function __construct($name) {
			parent::__construct($name, "The plugin could not be found");
		}
	}
	
	/**
	 * Exception class for detected exceptions while loading or executing
	 * plugins
	 * 
	 * This exception gets called if a plugin throws an exception while a plugin
	 * is loaded or while any internal call is made upon the plugin class.
	 */
	class PluginLoadError extends PluginError {
		/**
		 * @param string $name  The name of the plugin
		 * @param mixed  $error The exception class to examine
		 */
		public function __construct($name, $error) {
			$this->title = "An error occurred plugin: $name<br />\n";
			
			$this->message  = get_class($error) . ': ' . "<br />\n";
			$this->message .= '<pre>' . strval($error) . '</pre>';
		}
	}
	
	/**
	 * Exception base class for invalid plugins
	 */
	class PluginIntegrityError extends PluginError {}
	
	
	
	/**
	 * The basic layout for plugins
	 */
	interface PluginLayout {}
	
	/**
	 * The basic layout for the main plugin class
	 */
	interface PluginLayoutMain extends PluginLayout {
		/**
		 * @param Configuration $configuration The current configuration
		 * @param PluginLoader  $loader        The loading PluginLoader instance
		 */
		public function __construct($configuration, $loader);
		
		/**
		 * Give back in which contexts the plugin can be used
		 * 
		 * @return array
		 */
		public function supported();
		
		/**
		 * The the current version of the plugin
		 * 
		 * @return string
		 */
		public function get_version();
	
		/**
		 * Is it always required to include the main plugin file?
		 * 
		 * If this function does not return true then the contents of the main
		 * class will be cached and the main plugin file (plugin.php) will not
		 * be loaded the next time the plugin is used.
		 * 
		 * @return bool
		 */
		public function include_main();
	}
	
	/**
	 * The basic layout for all plugin implementations
	 * 
	 * If you want to use a different constructor definition then you should
	 * deliver from PluginLayout instead.
	 */
	interface PluginLayoutPlugin extends PluginLayout {
		/**
		 * @param Configuration $config  The current configuration
		 * @param mixed         $loader  The object loading this plugin
		 */
		public function __construct($config, $loader);
	}
	
	
	interface PluginLayoutPluginloader extends PluginLayout {
		/**
		 * @param Configuration $config
		 *        A configuration instance for this plugin
		 * @param PluginLoader  $loader
		 *        The object loading this plugin
		 * @param string        $name
		 *        The name of the plugin which is going to be loaded
		 * @param array         $cache_entry
		 *        The contents of the cache entry stored for this plugin
		 * 
		 * @throws PluginNotFoundError
		 */
		public function __construct(Configuration $config, PluginLoader $loader, $name, array $cache_entry);
		
		
		/**
		 * Return the current value of the cache entry of the plugin loaded
		 * 
		 * @return array
		 */
		public function get_cache_entry();
		
		
		/**
		 * Return the list of plugin types supported by the plugin loaded
		 * 
		 * @return array
		 */
		public function get_supported();
		
		
		/**
		 * Load and initalize a plugin class of the plugin currently loaded
		 * 
		 * @param string $type
		 *        The type of the plugin to load
		 * @param object $loader
		 *        The loaded of the plugin class
		 * @param bool   $initialize
		 *        Should the main plugin class be initialized?
		 * @param array  $arguments
		 *        All remaining arguments which should be passed to the
		 *        plugin class
		 * @return object
		 */
		public function load($type, $loader, $initialize, array $arguments);
	}
	
	
	
	
	
	
	/**
	 * Class which loads plugins
	 */
	class PluginLoader {
		/**
		 * An array contianing information about all plugins
		 * 
		 * @var array
		 */
		static protected $plugincache = null;
		
		
		
		/**
		 * Should the plugin classes provided by the plugin be initialized?
		 * 
		 * @var bool
		 */
		public $initialize = true;
		
		/**
		 * The plugin loader plugin which is used to load the requested plugin
		 * 
		 * @var PluginLayoutPluginloader
		 */
		protected $pluginloader;
		
		
		
		
		
		/**
		 * 
		 */
		public static function write_plugincache() {
			global $CONFIGURATION;
			$config = $CONFIGURATION;
			
			file_put_contents(
				"{$config->runtime()->fetch('*system.dir')}/plugins.psdf",
				serialize(self::$plugincache)
			);
		}
		
		
		
		
		
		/**
		 * @param string        $name   The name of the plugin to load
		 * @param Configuration $config The configuration object
		 * @param array         $path   A list of extra search paths of plugins
		 */
		public function __construct($name, $config=null, array $path=array()) {
			if(!($config instanceof Configuration)) {
				global $CONFIGURATION;
				$config = $CONFIGURATION;
			}
			
			// Make sure there is at least an empty array which can be passed
			// to the Plugin-object
			if(!$config->storage()->exists("plugin.cache.{$name}")) {
				$config->storage()->create("plugin.cache.{$name}", array());
			}
			
			report2(INFO, "Loading plugin: {$name}...");
			
			// Load the Pluginloader plugin to load this plugin class
			//TODO: Allow other Pluginloader plugins here
			try{
				$this->pluginloader = new FilePluginloaderPlugin(
					$config->main()->create_new('filesystem.'),
					$this,
					$name,
					$config->storage()->fetch("plugin.cache.{$name}")
				);
				
				// Update plugin cache for this plugin if necassery
				$config->storage()->update(
					"plugin.cache.{$name}",
					$this->pluginloader->get_cache_entry()
				);
			} catch(PluginNotFoundError $_e) {
				$this->pluginloader = null;
			}
		}
		
		
		/**
		 * Does the referenced plugin really exist?
		 * 
		 * @return bool
		 */
		public function exists() {
			return is_object($this->pluginloader);
		}
		
		
		/**
		 * Return a complete search path list
		 * 
		 * If an item of $search starts with ./ then it will be appended to all
		 * default search path items otherwise it will just be added to the list
		 * of search paths.
		 * 
		 * @param array $search An array of predefined search paths
		 * 
		 * @return array
		 */
		public static function get_search_path(array $search=array(), Configuration $config=null) {
			//TODO: Move into FilePluginloaderPlugin
			if(!is_object($config)) {
				global $CONFIGURATION;
				$config = $CONFIGURATION;
			}
			
			static $system_dir;
			if(is_null($system_dir)) {
				$system_dir = $config->runtime()->fetch('*system.dir');
			}
			
			$prefixpath = array("{$system_dir}/plugins");
			if($config->has_config('*plugin.directories')) {
				$prefixpath = array_merge(
				  $prefixpath,
				  (array) $config->config()->fetch('*plugin.directories')
				);
			}
			
			$path = $prefixpath;
			foreach((array) $search as $item) {
				// Add search path item to every plugin path
				if(strpos($item, './') === 0) {
					foreach($prefixpath as $prefix) {
						$path[] = $prefix . '/' . substr($item, 2);
					}
				} else {
					$path[] = $item;
				}
			}
			
			return $path;
		}
		
		
		/**
		 * List and filter plugins
		 * 
		 * List all plugins and filter them by type and naming prefix. 
		 * 
		 * @param string $type   Type to filter
		 * @param string $prefix A string with which the name should start
		 * @param array  $path   The search path for `self::get_search_path`
		 * 
		 * @return list
		 */
		public static function list_plugins($type=null, $prefix='', $path=array()) {
			//TODO: Move into FilePluginloaderPlugin
			$plugins = array();
			foreach(self::get_search_path($path) as $plugin_dir) {
				foreach(scandir($plugin_dir) as $plugin) {
					// Skip the current (.) and the parent (..) directory
					if($plugin == '.' || $plugin == '..')
						continue;
					
					// Skip plugins with invalid structures
					if(!file_exists("{$plugin_dir}/{$plugin}/plugin.php")) {
						continue;
					}
					
					// Check name starts with $prefix
					if($prefix && strpos($plugin, $prefix) !== 0) {
						continue;
					}
					
					// Check if type is supported by plugin
					if($type) {
						try {
							$self = new self($plugin);
						} catch(PluginError $_e) {
							continue;
						}
						
						if(!$self->is_supported($type)) {
							continue;
						}
					}
					
					$plugins[] = $plugin;
				}
			}
			
			return $plugins;
		}
		
		
		/**
		 * Load a single plugin class directly
		 * 
		 * @param string        $name   The name of the plugin to load
		 * @param string        $type   The type of plugin to load
		 * @param Configuration $config The current configuration object
		 * @param mixed         $loader The object invoking this plugin
		 * @param mixed         $...    Plugin-type specific arguments
		 * 
		 * @return mixed|false The loaded plugin object
		 */
		public static function load_plugin($name, $type, $config=null, $loader=null) {
			$loader = new self($name, $config);
			
			$args = array_merge(array($type), array_slice(func_get_args(), 3));
			return call_user_func_array(array($loader, 'load'), $args);
		}
		
		
		/**
		 * Try to load the plugin class
		 * 
		 * @param string $type   The type of plugin to load
		 * @param mixed  $loader The object which is loading the plugin
		 * @param mixed  ...     Plugin-type specific arguments (see interfaces)
		 *
		 * @return mixed The loaded plugin object or false on error
		 */
		public function load($type, $loader=null) {
			// Don't continue if the plugin does not support the type specified
			if(!$this->is_supported($type)) {
				return false;
			}
			
			return $this->pluginloader->load(
				$type,
				$loader,
				$this->initialize,
				array_slice(func_get_args(), 2)
			);
		}
		
		
		/**
		 * Are the plugin classes generated by the plugin loader going to be
		 * initialized?
		 * 
		 * @return bool
		 */
		public function is_initializing() {
			return $this->initialize;
		}
		
		
		/**
		 * Set if classes generated by the plugin loader should be initialized
		 * 
		 * @param bool $initialize
		 *        Should the generated classes be initialized?
		 */
		public function set_initializing($initialze=true) {
			$this->initialize = $initialze;
		}
		
		
		/**
		 * Check if a certain type is supported
		 * 
		 * @return bool
		 */
		public function is_supported($type) {
			if($this->exists()) {
				return in_array(strtolower($type), $this->get_support());
			} else {
				return false;
			}
		}
		
		
		/**
		 * Get a list of supported types by the plugin
		 * 
		 * @return array
		 */
		public function get_support() {
			if(!$this->exists()) {
				return array();
			}
			
			/*$supported = array();
			foreach($this->pluginloader->get_supported() as $support) {
				$_merged = '';
				
				foreach(explode('_', $support) as $part) {
					if(!$_merged) {
						$_merged = $part;
					} else {
						$_merged = "{$_merged}_{$part}";
					}
					
					if(!in_array($_merged, $supported)) {
						$supported[] = $_merged;
					}
				}
			}
			
			return $supported;*/
			return $this->pluginloader->get_supported();
		}
	}
	
	
	
	/**
	 * Include an aCMS component (plugin API function)
	 * 
	 * The component path should be relative to the aCMS-base directory and will
	 * be included like a normal PHP file.
	 * 
	 * @uses include_once
	 * @access public
	 * @param string $path The component to include
	 */
	function include_component($path) {
		// Include the component fileuseful for plugins
		$rv = include_once(__DIR__ . '/' . "../$path");
		// Export global variables
		$GLOBALS += get_defined_vars();
		// Return the include return value
		return $rv;
	}
	
	
	/**
	 * Require an aCMS component (plugin API function)
	 * 
	 * The component path should be relative to the aCMS-base directory and will
	 * be required like a normal PHP file.
	 * 
	 * @uses require_once
	 * @access public
	 * @param string $path The component to require
	 */
	function require_component($path) {
		// Require the component file
		$rv = require_once(__DIR__ . '/' . "../$path");
		// Export global variables
		$GLOBALS += get_defined_vars();
		// Return the require return value
		return $rv;
	}
	
	
	require_once(dirname(__FILE__) . '/' . "FilePluginloaderPlugin.class.php");
