<?php
	require_once(dirname(__FILE__) . '/' . "../report.inc.php");
	require_once(dirname(__FILE__) . '/' . "ConfigurationFileParser.class.php");
	require_once(dirname(__FILE__) . '/' . "PluginLoader.class.php");
	
	
	
	
	
	/**
	 * Exception class for undefined plugin classes
	 * 
	 * This exception gets thrown if a plugin does not define a main class or a
	 * as- supported-claimed plugin class.
	 */
	class PluginClassIntegrityError extends PluginIntegrityError {
		/**
		 * @param string $name The name of the plugin
		 * @param string $type The used type of the plugin
		 */
		function __construct($name, $type) {
			$this->message = 
				"The plugin \"$name\" does not implement a plugin class for " .
				"the type \"$type\"";
		}
	}
	
	/**
	 * Exception class for invalid plugin classes
	 * 
	 * This exception gets thrown if a plugin does not implement an interface
	 * which it is suppose to be used.
	 */
	class PluginInterfaceIntegrityError extends PluginIntegrityError {
		/**
		 * @param string $name      The name of the plugin
		 * @param string $class     The name of the class
		 * @param string $interface The name of the interface
		 */
		public function __construct($name, $class, $interface) {
			$this->message =
				"The plugin \"$name\" has an invalid plugin class \"$class\" " .
				"which does not implement the interface \"$interface\"";
		}
	}
	
	
	
	
	
	/**
	 * A class which manages all file system-based plugins
	 */
	class FilePluginloaderPlugin implements PluginLayoutPluginloader {
		/**
		 * The plugin cache data of this plugin
		 * 
		 * @var array
		 */
		protected $cache = array();
		
		/**
		 * Should be plugin cache be passed to PluginLoader for cross-session
		 * caching?
		 * 
		 * @var bool
		 */
		public $cache_cached = true;
		
		/**
		 * The name of this plugin
		 * 
		 * @var string
		 */
		protected $name = '';
		
		
		
		/**
		 * @param Configuration $config
		 *        A configuration instance (only required for preconfig calls!)
		 * @param PluginLoader  $loader
		 *        The object loading this plugin
		 * @param string        $name
		 *        The name of the plugin which is going to be loaded
		 * @param array         $cache_entry
		 *        The contents of the cache entry stored for this plugin
		 * 
		 * @throws PluginNotFoundError
		 */
		public function __construct(Configuration $config, PluginLoader $loader, $name, array $cache_entry) {
			$this->config = $config->main();
			$this->name   = strtolower($name);
			$this->cache  = $cache_entry;
			
			// Throw an exception if the plugin does not exist
			if(!$this->get_directory_path()) {
				throw new PluginNotFoundError($name);
			}
			
			// Make sure the cache we are going to use is valid
			$this->fix_cache();
		}
		
		
		/**
		 * Return the data stored in the plugin cache or just an empty array
		 * if plugin cache caching was disabled
		 * 
		 * @return array
		 */
		public function get_cache_entry() {
			if($this->cache_cached) {
				return $this->cache;
			} else {
				return array();
			}
		}
		
		
		/**
		 * Return the base directory of the plugin
		 * 
		 * @return string|null
		 */
		public function get_directory_path(array $path=array()) {
			// Search for plugin in file system
			foreach(PluginLoader::get_search_path($path, $this->config) as $filepath) {
				if(is_file("{$filepath}/{$this->name}/plugin.php")) {
					return "{$filepath}/{$this->name}";
				}
			}
			
			return null;
		}
		
		
		/**
		 * Return the list of plugin types supported by the plugin
		 * 
		 * @return array
		 */
		public function get_supported() {
			return $this->cache['supported'];
		}
		
		
		/**
		 * Return a reference to the main class object provided by this class
		 * 
		 * @return object
		 */
		public function find_main_class() {
			// Initialize main class
			try {
				require_once($this->get_directory_path() . '/' . "plugin.php");
			} catch(Exception $e) {
				throw new PluginLoadError($this->name, $e);
				return;
			}

			$main = "{$this->cache['classname_prefix']}Main";
			if(!class_exists($main)) {
				throw new PluginClassIntegrityError($this->name, 'main');
				return;
			}

			$reflection = new ReflectionClass($main);
			if(!$reflection->implementsInterface('PluginLayoutMain')) {
				throw new PluginInterfaceIntegrityError(
				  $this->name, $main, 'PluginLayoutMain');
				return null;
			}

			try {
				return new $main($this->config, $this);
			} catch(Exception $e) {
				throw new PluginLoadError($this->name, $e);
				return null;
			}
		}


		
		/**
		 * Generate the prefix which should be used in front of all class names
		 * which might be registered by this plugin
		 * 
		 * @return string
		 */
		public function find_class_prefix() {
			// Generate the prefix to use in front of classes provided by
			// sub-classes of this class
			$uppercase    = true;
			$class_prefix = '';
			foreach(str_split($this->name) as $char) {
				// If a seperating charactor is processed switch to uppercase
				// character mode
				if($char == '_' || $char == ' ' || $char == '/') {
					$uppercase = true;
				// If we are in uppercase character mode write one character in
				// in upper case and then switch to lower case
				} elseif($uppercase) {
					$class_prefix .= strtoupper($char);
					$uppercase = false;
				// Just write the character lower case
				} else {
					$class_prefix .= strtolower($char);
				}
			}
			
			return $class_prefix;
		}
		
		
		/**
		 * Check the cache entry which was provided and update missing entries
		 */
		public function fix_cache() {
			$last_change = filemtime($this->get_directory_path());
			if(!isset($this->cache['last_changed'])
			|| $this->cache['last_changed'] < $last_change) {
				// Completely renew cache if any of the files within the plugin
				// directory have changed
				$this->cache = array();
				$this->cache['last_changed'] = $last_change;
			}
			
			// Do configuration processing if not already done
			if(!isset($this->cache['config_processed'])
			|| !$this->cache['config_processed']) {
				$this->cache['config_processed'] = $this->process_config();
			}
			
			// Cache the class prefix used
			if(!isset($this->cache['classname_prefix'])
			|| !is_string($this->cache['classname_prefix'])) {
				$this->cache['classname_prefix'] = $this->find_class_prefix();
			}
			
			// Load the main class file if the plugin's main class requested
			// it
			$main = null;
			if(!isset($this->cache['include_main'])
			|| $this->cache['include_main']) {
				if(!is_object($main)) {
					$main = $this->find_main_class();
				}
				
				$this->cache['include_main'] = $main->include_main();
			}
			
			// Cache the list of supported plugin types
			if(!isset($this->cache['supported'])
			|| !is_array($this->cache['supported'])) {
				if(!is_object($main)) {
					$main = $this->find_main_class();
				}
				
				$this->cache['supported'] = $main->supported();
			}
			
		}
		
		
		/**
		 * Load and initalize a plugin class of the plugin currently loaded
		 * 
		 * @param string $type
		 *        The type of the plugin to load
		 * @param object $loader
		 *        The loaded of the plugin class
		 * @param bool   $initialize
		 *        Should the main plugin class be initialized?
		 * @param array  $arguments
		 *        All remaining arguments which should be passed to the
		 *        plugin class
		 * @return object|string
		 */
		public function load($type, $loader, $initialize, array $arguments) {
			$type_parts = array_reverse(explode('_', strtolower($type)));
			
			// Try to find the most suitable interface for the plugin type
			$interface = '';
			for($x=0, $max=count($type_parts); $x <= $max; $x++) {
				// Assemble the interface name to check for
				//  $x=0: PluginLayoutFilterRaw
				//  $x=1: PluginLayoutFilter
				//  $x=2: PluginLayout
				$interface = 'PluginLayout';
				for($y=count($type_parts)-1; $y >= $x; $y--){
					$interface .= ucfirst($type_parts[$y]);
				}
				
				// Check if the interface really exists
				if(interface_exists($interface)) {
					break;
				}
			}
			
			
			// Generate the filename of the file to load in order to provide
			// the expected type
			$filepath = "{$this->get_directory_path()}/plugin";
			for($x=count($type_parts)-1; $x >= 0; $x--) {
				$filepath .= ".{$type_parts[$x]}";
			}
			$filepath .= '.php';
			
			// Load the file providing the main class
			require_once($filepath);
			
			
			// Generate the class name of the plugin class providing the plugins
			// requested functions
			$classname = $this->cache['classname_prefix'];
			for($x=0, $max=count($type_parts); $x < $max; $x++) {
				$classname .= ucfirst($type_parts[$x]);
			}
			$classname .= 'Plugin';
			
			// Check if a plugin class could be found
			if(!class_exists($classname)) {
				throw new PluginClassIntegrityError($this->name, $type);
				return false;
			}
			
			// Create plugin sub-configuration instance
			$config = $this->config->create_new("{$this->name}.");
			
			// Initialize reflection of the plugin class
			$reflection = new ReflectionClass($classname);
			
			// Check if the plugin class implements the required interface
			if(!$reflection->implementsInterface($interface)) {
				throw new PluginInterfaceIntegrityError(
				  $this->name, $classname, $interface
				);
				return false;
			}
			
			//try {
				if($initialize) {
					$arguments = array_merge(
						array($config, $loader), $arguments
					);
					//$arguments = array_merge(array(null, null), $arguments);

					// Initialize the class via the `initialize`-method if it is
					// provided by the plugin class
					if($reflection->hasMethod('initialize')) {
						$object = $reflection->newInstance();
						
						// Call the initialize method of the new plugin object
						$init = new ReflectionMethod($object, 'initialize');
						$init->invokeArgs($object, $arguments);
						
						return $object;
					// Go the normal way of passing the arguments to the
					// constructor if there is no `initialize`-method
					} else {
						return $reflection->newInstanceArgs($arguments);
					}
				} else {
					// Set the new classes config_static attribute if possible
					$prop = new ReflectionProperty($classname, 'config_static');
					if($prop->isStatic()) {
						$prop->setValue($config);
					}
					
					// Set the new classes loader_static attribute if possible
					$prop = new ReflectionProperty($classname, 'loader_static');
					if($prop->isStatic()) {
						$prop->setValue($loader);
					}
					
					return $classname;
				}
			//} catch(Exception $error) {
			//	throw new PluginLoadError($this->name, $error);
			//	return false;
			//}
		}
		
		
		public function process_config_item($name, $data) {
			// Skip invalid items
			if(!is_array($data) || count($data) != 2) {
				continue;
			}
			
			// Read and find all necacery data from the item set
			report2(DEBUG,
				" * Found new item: {$name}"
			);

			// Disable plugin data caching on developer request
			if($name == "{$this->name}.no_cache") {
				$this->cache_cached = false;
			// Check if the item doesn't already exist before continuing
			} elseif(!$this->config->config()->exists($name)) {
				// Create the name of the constant which should hold
				// the integer representing the type specified
				$type_const = 
					"ConfigurationHelperConfig::TYPE_{$data[0]}";

				// Create new configuration element if the type is valid
				// and cheer!
				if(defined($type_const)) {
					// Actually register the new value
					$this->config->config()->create(
						$name, $data[1], constant($type_const)
					);
					
					report2(INFO,
					  "Registered new plugin configuration value: " .
					  "{$name}, {$data[1]}, {$data[0]}"
					);
				} else {
					report2(WARNING,
					  "Invalid plugin configuration type: {$data[0]}! ".
					  "(Plugin: {$this->name}, Name: {$name})"
					);
				}
			}
		}
		
		
		public function process_config() {
			$config_path = "{$this->get_directory_path()}/config.json";
			
			// Don't continue if there is config.json configuration data file
			if(!is_readable($config_path)) {
				report2(INFO, "No config.json found! (Plugin: {$this->name})");
				return 1;
			}
			
			// Save the time when parsing started for the final message
			$time1 = microtime(true);
			
			report2(DEBUG, "Parsing plugin configuration... {$config_path}");
			$time2 = microtime(true);
			
			// Strip comments from JSON file
			$contents = "";
			foreach(explode("\n", file_get_contents($config_path)) as $line) {
				if(substr(trim($line), 0, 2) != '//') {
					$contents .= "{$line}\n";
				}
			}
			
			$config_data = json_decode($contents);
			if(!is_object($config_data)) {
				switch(json_last_error()) {
					case JSON_ERROR_DEPTH:
						$message = "The maximum stack depth has been exceeded.";
					break;
					case JSON_ERROR_STATE_MISMATCH:
						$message = "Underflow or modes mismatch.";
					break;
					case JSON_ERROR_CTRL_CHAR:
						$message  = "Control character error, ";
						$message .= "possibly incorrectly encoded.";
					break;
					case JSON_ERROR_SYNTAX:
						$message = "Syntax error.";
					break;
					default:
						$message = "Unkown error (" . json_last_error() . ").";
				}
				
				report2(WARNING,
				  "An error occured during parsing of plugin configuration! " .
				  "{$message} (Plugin: {$this->name}, Path: {$config_path})"
				);
				return false;
			} else {
				$time = microtime(true) - $time2;
				report2(DEBUG, "Parsing successful! ({$time}s)");
			}
			
			report2(DEBUG, "Reading and populating result data...");
			foreach((array) $config_data as $name => $data) {
				// Skip non PHP Array entries
				if(!is_array($data) && !is_object($data)) {
					continue;
				}
				
				if(is_object($data)) {
					// Determine the section name
					if(substr($name, 0, 1) == '*') {
						$section = substr($name, 1);
					} else {
						$section = "{$this->name}.{$name}";
					}
					
					foreach((array) $data as $item => $content) {
						// Process normal item
						$this->process_config_item(
							"{$section}.{$item}", $content
						);
					}
				} else {
					// Process normal item
					$this->process_config_item(
						"{$this->name}.{$name}", $data
					);
				}
			}
			
			$time = microtime(true)- $time1;
			report2(INFO,
			  "Successfully processed plugin json configuration file! " .
			  "(Plugin: {$this->name}, Path: {$config_path}, Time: {$time}s)"
			);
			return true;
		}
		
		
		/**
		 * Parse the contents of the plugin's config.ini and populate its
		 * contents
		 * 
		 * @uses report2
		 * 
		 * @return boolean Success?
		 */
		/*public function process_config_data() {
			if($this->process_config_json() === true) {
				return true;
			}
			
			// Save the time when parsing started for the final message
			$time1 = microtime(true);
			
			$config_path = "{$this->get_directory_path()}/config.ini";
			
			// Don't continue if there is config.ini configuration data file
			if(!is_readable($config_path)) {
				report2(INFO, "No config.ini found! (Plugin: {$this->name})");
				return true;
			}
			
			report2(DEBUG, "Parsing plugin configuration... {$config_path}");
			$time2 = microtime(true);
			try {
				$parser      = new ConfigurationFileParser();
				$config_file = fopen($config_path, 'r');
				$config_data = $parser->process($config_file);
				fclose($config_file);
			} catch(ConfigurationFileParserError $error) {
				report2(WARNING,
				  "An error occured during parsing of plugin configuration: " .
				  "{$error->message} (Plugin: {$this->name}, " .
				  "Path: {$config_path})"
				);
				return false;
			}
			$time = microtime(true) - $time2;
			report2(DEBUG, "Parsing successful! ({$time}s)");
			
			report2(DEBUG, "Reading and populating result data...");
			foreach($config_data as $section => $items) {
				// Read and interpert the section field given
				if($section == 'DEFAULT') {
					$section = $this->name;
				} elseif(substr($section, 0, 1) == '*') {
					$section = substr($section, 1);
				} else {
					$section = "{$this->name}.{$section}";
				}
				report2(DEBUG, " * Found new section: {$section}");
				
				foreach($items as $name => $data) {
					// Skip invalid items
					if(!is_array($data) || count($data) != 2) {
						continue;
					}
					
					// Read and find all necacery data from the item set
					$this->process_config_item("{$section}.{$name}", $data);
				}
			}
			
			$time = microtime(true)- $time1;
			report2(INFO,
			  "Successfully processed plugin config configuration file! " .
			  "(Plugin: {$this->name}, Path: {$config_path}, Time: {$time}s)"
			);
			return true;
		}*/
	}
