<?php
	/**
	 * @package aCMS
	 */
	require_once(__DIR__ . '/' . "../AOF/AOFObject.class.php");
	require_once(__DIR__ . '/' . "../AOF/AOFError.class.php");
	require_once(__DIR__ . '/' . "../AOF/AOFErrorTemplate.class.php");
	
	require_once(__DIR__ . '/' . "ConfigurationConnection.class.php");
	require_once(__DIR__ . '/' . "PluginLoader.class.php");
	
	require_once(__DIR__ . '/' . "../report.inc.php");
	
	
	
	/**
	 * Base exception class for all configuration errors
	 */
	class ConfigurationError extends AOFError {
		/**
		 * The title of the error page
		 * 
		 * @var string
		 */
		protected $title = 'An configuration error occurred ($CLASS)';
		
		
		public $helper = "";
		public $key    = "";
	}
	
	
	/**
	 * Exception class thrown if a requested configuration key could not be
	 * found
	 */
	class ConfigurationKeyNotFoundError extends ConfigurationError {
		/**
		 * @param string $helper
		 *        The name of the configuration helper component used
		 * @param string $key
		 *        The name of the configuration which couldn't be found
		 */
		public function __construct($helper, $key) {
			$this->helper = $helper;
			$this->key    = $key;
			
			parent::__construct(
					"A configuration key was requested but cannot be found: " .
					"{$helper}: {$key}"
			);
		}
	}
	
	/**
	 * PDO database configuration error
	 */
	class ConfigurationPDOError extends ConfigurationError {
		public function __construct($errorinfo, $sql) {
			parent::__construct("{$errorinfo[2]} ({$errorinfo[0]}/{$errorinfo[1]})");
			$this->infos = "SQL command executed: <pre>{$sql}</pre>";
		}
	}
	
	/**
	 * 
	 */
	class CacheDirectoryError extends ConfigurationError {
		public function __construct() {
			$this->message = "The cache directory <em>cache.directory</em> was not " .
				"set in the configuration.";
			
			$this->solutions = array("Set that variable in the configuration");
		}
	}
	
	/**
	 * Exception class for storage file related errors
	 */
	class StorageFileError extends ConfigurationError {
		/**
		 * The template of this exception class
		 * 
		 * @var string
		 */
		protected $tpl = 'AOFFileErrorTemplate';
		
		
		/**
		 * @param string $filepath The path of the storage file
		 */
		public function __construct($filepath=null) {
			$this->solutions = $this->tpl_solutions($filepath, 'r+');
			$this->message   = sprintf(
				$this->tpl_message($filepath, 'r+'), 'storage');
		}
	}
	
	
	
	/**
	 * Plugin layout for all configuration plugins
	 */
	interface PluginLayoutConfiguration extends PluginLayout {
		/**
		 * @param Configuration     $config The current configuration object
		 * @param mixed             $loader The loading object
		 * @param ConfigurationData $data   The connection data
		 */
		public function __construct($config, $loader, $data);
		
		
		/**
		 * Create a new configuration item
		 * 
		 * At least one of $value or $type must be present
		 * 
		 * @param string $section
		 *        The name of the configuration section
		 * @param string $name
		 *        The name of the configuration item
		 * @param mixed  $value
		 *        The new value to set for the configuration item
		 * @param string $type
		 *        The type of the new item (Default: Auto-detect $value)
		 * @return bool Was the item successfully created?
		 */
		public function item_create($section, $name, $value, $type);
		
		
		/**
		 * Check if a configuration item exists
		 * 
		 * @param string $section
		 *        The name of the section
		 * @param string $name
		 *        The name of the configuration item
		 * @return bool
		 */
		public function item_exists($section, $name);
		
		
		/**
		 * Fetch a configuration item and return its value
		 * 
		 * @param string $section
		 *        The name of the section
		 * @param string $name
		 *        The name of the configuration item
		 * @return bool|float|integer|string|null
		 */
		public function item_fetch($section, $name);
		
		
		/**
		 * Change/update the value of a configuration item
		 * 
		 * @param string $section
		 *        The name of the section
		 * @param string $name
		 *        The name of the configuration item
		 * @param mixed  $value
		 *        The value to write into the database
		 * @return bool Where any changes made?
		 */
		public function item_update($section, $name, $value);
		
		
		/**
		 * Fetch all configuration items within a given section
		 * 
		 * Output format: array($name => $value, ...)
		 * 
		 * @param string $section
		 *        The name of the section which items should be fetched
		 * @return array
		 */
		public function section_fetch($section);
		
		
		/**
		 * Fetch all configuration items
		 * 
		 * Output format: array($section => array($name => $value, ...), ...)
		 * 
		 * @return array
		 */
		public function table_fetch();
		
		
		/**
		 * Give back the time of last modification of the configuration or false
		 * if that value could not be determined
		 * 
		 * @return integer|false
		 */
		public function mtime();
	}
	
	
	
	/**
	 * Abstract class 
	 */
	abstract class PluginHelperConfigurationPDO implements PluginLayoutConfiguration {
		/**
		 * SQL command to create a new configuration data table in the database
		 */
		const SQL_TABLE_CREATE = '';
		
		/**
		 * SQL command to fetch all configuration data entries from the table
		 */
		const SQL_TABLE_FETCH = '';
		
		/**
		 * SQL command to fetch all configuration data entries of a certain
		 * section from the table
		 */
		const SQL_SECTION_FETCH = '';
		
		/**
		 * SQL command to create/insert a new configuration item into the
		 * configuration table
		 */
		const SQL_ITEM_CREATE = '';
		
		/**
		 * SQL command to check if a configuration item exists
		 */
		const SQL_ITEM_EXISTS = '';
		
		/**
		 * SQL command to fetch a configuration item limited by its name and
		 * section from the data
		 * 
		 * You might want to use the following SQL extensions in your
		 * SQL command: LIMIT = 1
		 */
		const SQL_ITEM_FETCH = '';
		
		/**
		 * SQL command to change to value of a configuration item inside of the
		 * configuration table
		 */
		const SQL_ITEM_UPDATE = '';
		
		
		
		/**
		 * An array containing all type names as keys and the database type
		 * names as values
		 * 
		 * @var array
		 */
		protected $type_map = array(
		  ConfigurationHelperConfig::TYPE_SI_BOOLEAN => "sb",
		  ConfigurationHelperConfig::TYPE_SI_INTEGER => "si",
		  ConfigurationHelperConfig::TYPE_SI_FLOAT   => "sf",
		  ConfigurationHelperConfig::TYPE_SI_STRING  => "ss",
		  ConfigurationHelperConfig::TYPE_MU_BOOLEAN => "mb",
		  ConfigurationHelperConfig::TYPE_MU_INTEGER => "mi",
		  ConfigurationHelperConfig::TYPE_MU_FLOAT   => "mf",
		  ConfigurationHelperConfig::TYPE_MU_STRING  => "ms",
		);
		
		
		
		/**
		 * A PDO database connection which should be set during class
		 * construction
		 * 
		 * @var PDO
		 */
		protected $db;
		
		/**
		 * An object providing all information about the connection
		 * 
		 * @var ConfigurationConnection
		 */
		protected $data;
		
		
		
		
		protected function execute($name, $args=array()) {
			// Quote all statements manually
			//  (since table name quoting is not supported natively)
			$sql = constant(get_class($this).'::'."SQL_{$name}");
			foreach($args as $name => $value) {
				$sql = str_replace(
				  $name, substr($this->db->quote($value), 1, -1), $sql
				);
			}
			
			// Submit SQL to database and return response statement
			$statement = $this->db->query($sql);
			if($statement) {
				return $statement;
			} else {
				throw new ConfigurationPDOError($this->db->errorInfo(), $sql);
			}
		}
		
		
		
		/**
		 * Construction helper setting up database connection and creating the
		 * table if it does not already exist
		 * 
		 * @param array  $data
		 *        The connection information to connect to the database
		 * @param string $dsn
		 *        The PDO database DSN to use to connect
		 */
		protected function construct_helper($data, $dsn) {
			$this->data = $data;
			
			try {
				// Connect to the database server
				$this->db = new PDO(
						$dsn,
						$data->username,
						$data->password,
						array(PDO::ATTR_PERSISTENT => true)
				);
				// Silence PDO and handle all errors ourselves
				$this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_SILENT);
			} catch(PDOException $error) {
				// Rethrow errors as ConfigurationErrors
				//  (and generate error info on-the-fly)
				$p = explode(']', $error->getMessage(), 3);
				
				$message = (count($p) > 2) ? $p[2] : "";
				
				throw new ConfigurationPDOError(
						array(substr($p[0], 9), $error->getCode(), $message),
						""
				);
			}
			
			// Create the table in the database (if it not already exists)
			try {
				$this->execute('TABLE_CREATE', array('$table' => $data->table));
			} catch(ConfigurationPDOError $error) {}
		}
		
		
		/**
		 * Create a new configuration item in the database
		 * 
		 * At least one of $value or $type must be present
		 * 
		 * @param string $section
		 *        The name of the configuration section
		 * @param string $name
		 *        The name of the configuration item
		 * @param mixed  $value
		 *        The new value to set for the configuration item
		 * @throws DBConfgurationError
		 */
		public function item_create($section, $name, $value, $type=null) {
			// Check if a valid type was given
			if(!isset($this->type_map[$type])) {
				// Auto-detect type
				$type = ConfigurationHelperConfig::value_find_type($value);
				// Exit if auto-detection failed
				if(!isset($this->type_map[$type])) {
					return false;
				}
			}
			
			// Create the new database item
			$this->execute('ITEM_CREATE', array(
			  '$table'   => $this->data->table,
			  '$section' => $section,
			  '$name'    => $name,
			  '$value'   => $value,
			  '$type'    => $this->type_map[$type],
			));
		}
		
		
		/**
		 * Check if a configuration item exists
		 * 
		 * @param string $section
		 *        The name of the section
		 * @param string $name
		 *        The name of the configuration item
		 * @return bool
		 */
		public function item_exists($section, $name) {
			// Tell database to count the number of rows with this
			// section and name pair
			$statement = $this->execute('ITEM_EXISTS', array(
			  '$table'   => $this->data->table,
			  '$section' => $section,
			  '$name'    => $name
			));
			
			// Get the row number
			$row = $statement->fetch(PDO::FETCH_NUM);
			
			// Return if there any row that matched the pattern
			return ((int) $row[0]) > 0;
		}
		
		
		/**
		 * Fetch a configuration item and return its value
		 * 
		 * @param string $section
		 *        The name of the section
		 * @param string $name
		 *        The name of the configuration item
		 * @throws DBConfgurationError
		 * @return bool|float|integer|string|null
		 */
		public function item_fetch($section, $name) {
			// Tell database to fetch row
			$statement = $this->execute('ITEM_FETCH', array(
			  '$table'   => $this->data->table,
			  '$section' => $section,
			  '$name'    => $name
			));
			
			// Retrieve the row from the database table
			if(!($row=$statement->fetch(PDO::FETCH_LAZY))) {
				return null;
			}
			
			// Return the resulting value and type
			return array(
			  'value' => $row['value'],
			  'type'  => array_search($row['type'], $this->type_map)
			);
		}
		
		
		/**
		 * Change/update the value of a configuration item
		 * 
		 * @param string $section
		 *        The name of the section
		 * @param string $name
		 *        The name of the configuration item
		 * @param mixed  $value
		 *        The value to write into the database
		 * @throws DBConfgurationError
		 * @return bool Where any changes made?
		 */
		public function item_update($section, $name, $value) {
			// Update the database item
			$statement = $this->execute('ITEM_UPDATE', array(
			  '$table'   => $this->data->table,
			  '$section' => $section,
			  '$name'    => $name,
			  '$value'   => $value,
			));
			
			return $statement->rowCount() > 0;
		}
		
		
		/**
		 * Fetch all configuration items within a given section and return all
		 * found items as key/value array
		 * 
		 * array($name => $value, ...)
		 * 
		 * @param string $section
		 *        The name of the section which items should be fetched
		 * @return array
		 */
		public function section_fetch($section) {
			// Tell database to fetch section
			$statement = $this->execute('SECTION_FETCH', array(
			  '$table'   => $this->data->table,
			  '$section' => $section
			));
			
			$section = array();
			// Retrieve a row from the database table
			while($row=$statement->fetch(PDO::FETCH_LAZY)) {
				$section[$row['name']] = array(
				  'value' => $row['value'],
				  'type'  => array_search($row['type'], $this->type_map)
				);
			}
			
			return $section;
		}
		
		
		/**
		 * Fetch all configuration items within the configuration table and
		 * return them as key=>key/value array
		 * 
		 * array($section => array($name => $value, ...), ...)
		 * 
		 * @return array
		 */
		public function table_fetch() {
			// Tell database to fetch table
			$statement = $this->execute('TABLE_FETCH', array(
			  '$table' => $this->data->table
			));
			
			$table = array();
			// Retrieve a row from the database table
			while($row=$statement->fetch(PDO::FETCH_LAZY)) {
				// Create row entry if it does not already exist
				if(!isset($table[$row["section"]])) {
					$table[$row["section"]] = array();
				}
				
				$table[$row["section"]][$row['name']] = array(
				  'value' => $row['value'],
				  'type'  => array_search($row['type'], $this->type_map)
				);
			}
			
			return $table;
		}
		
		
		/**
		 * Get the last modification time of the database
		 * 
		 * CURRENTLY UNIMPLEMENTED
		 * 
		 * @return false
		 */
		public function mtime() {
			return false;
		}
	}
	
	
	
	/**
	 * Base class for all Configuration helpers
	 * 
	 * @internal
	 */
	abstract class ConfigurationHelper extends AOFObject implements arrayaccess {
		/**
		 * @alias fetch
		 */
		public function get($key) {
			return $this->fetch($key);
		}
		
		
		/**
		 * @alias fetch
		 */
		public function offsetGet($offset) {
			return $this->fetch($offset);
		}
		
		
		/**
		 * Fetch an item from your configuration
		 * 
		 * This function should throw an ConfigurationKeyNotFoundError exception
		 * if the key fetched does not exist.
		 * 
		 * @throws ConfigurationKeyNotFoundError
		 * 
		 * @param string $key
		 *        The name of the item which should be retrieved
		 * @return mixed
		 */
		abstract public function fetch($key);
		
		
		
		/**
		 * Fetch all stored items from your configuration
		 * 
		 * This should return an key/value array where the keys are the
		 * configuration keys and the values are the data entries.
		 * 
		 * @return array
		 */
		abstract public function fetch_all();
		
		
		
		/**
		 * @alias exists
		 */
		public function has($key) {
			return $this->exists($key);
		}
		
		
		/**
		 * @alias exists
		 */
		public function offsetExists($offset) {
			return $this->exists($offset);
		}
		
		
		
		/**
		 * @alias update
		 */
		public function put($key, $value) {
			return $this->update($key, $value);
		}
		
		
		/**
		 * @alias update
		 */
		public function set($key, $value) {
			return $this->update($key, $value);
		}
		
		
		/**
		 * @alias update
		 */
		public function offsetSet($offset, $value) {
			return $this->update($offset, $value);
		}
		
		
		
		/**
		 * @alias remove
		 */
		public function delete($key) {
			return $this->remove($key);
		}
		
		
		/**
		 * @alias remove
		 */
		public function del($key) {
			return $this->remove($key);
		}
		
		
		/**
		 * @alias remove
		 */
		public function offsetUnset($offset) {
			return $this->remove($offset);
		}
	}
	
	
	/**
	 * A class that wraps around a configuration helper class and allows
	 * modifiers, like prefixes, to be used when redirecting each call
	 * 
	 * This class is currently only used for applying prefixes.
	 * 
	 * @internal
	 */
	class ConfigurationHelperproxy extends ConfigurationHelper {
		/**
		 * The object to redirect all calls to
		 * 
		 * @var object
		 */
		protected $main;
		
		/**
		 * A prefix to use in front of each call
		 * 
		 * @var string
		 */
		protected $prefix = '';
		
		
		
		/**
		 * @param ConfigurationHelper  $main
		 *        The object that the calls should be redirected too
		 * @param string              [$prefix='']
		 *        A string to use in front of every key name redirected
		 */
		public function __construct(ConfigurationHelper $main, $prefix='') {
			$this->main   = $main;
			$this->prefix = $prefix;
		}
		
		
		/**
		 * Return the prefix used with this helper proxy
		 * 
		 * @return string
		 */
		public function get_prefix() {
			return $this->prefix;
		}
		
		
		/**
		 * Return the main configuration helper used in this prefix
		 * 
		 * @return ConfigurationHelper
		 */
		public function get_main() {
			return $this->main;
		}
		
		
		/**
		 * Return the value of an item
		 * 
		 * @param string $key
		 *        The name of the item to fetch from the main object
		 * @return mixed|null
		 */
		public function fetch($key) {
			
			// Pass keyname to the main class unmodified if it has the global
			// prefix
			if(substr($key, 0, 1) != '*') {
				$key = "{$this->prefix}{$key}";
			}
			
			return $this->main->fetch($key);
		}
		
		
		/**
		 * Return all configuration values
		 * 
		 * @return array
		 */
		public function fetch_all() {
			return $this->main->fetch_all();
		}
		
		
		/**
		 * Check if an item exists
		 * 
		 * @param string $key
		 *        The name of the item to check for
		 * @return bool
		 */
		public function exists($key) {
			// Pass keyname to the main class unmodified if it has the global
			// prefix
			if(substr($key, 0, 1) != '*') {
				$key = "{$this->prefix}{$key}";
			}
			
			return $this->main->exists($key);
		}
		
		
		/**
		 * Change the value of an item
		 * 
		 * @param string $key
		 *        The name of the item to change
		 * @param mixed  $value
		 *        The new value of the item
		 * @return bool Success?
		 */
		public function update($key, $value) {
			// Pass keyname to the main class unmodified if it has the global
			// prefix
			if(substr($key, 0, 1) != '*') {
				$key = "{$this->prefix}{$key}";
			}
			
			return $this->main->update($key, $value);
		}
		
		
		/**
		 * Create a new item
		 * 
		 * @param string $key
		 *        The name of the new item
		 * @param mixed  $value
		 *        The value of the new item
		 * @param mixed  $arg1
		 *        Optional parameter
		 * @param mxed   $arg2
		 *        Optional parameter
		 * @return bool Success?
		 */
		public function create($key, $value, $arg1=null, $arg2=null) {
			// Pass keyname to the main class unmodified if it has the global
			// prefix
			if(substr($key, 0, 1) != '*') {
				$key = "{$this->prefix}{$key}";
			}
			
			return $this->main->create($key, $value, $arg1, $arg2);
		}
		
		
		/**
		 * Remove an item
		 * 
		 * @param string $key
		 *        The name of the runtime item to delete
		 */
		public function remove($key) {
			// Pass keyname to the main class unmodified if it has the global
			// prefix
			if(substr($key, 0, 1) != '*') {
				$key = "{$this->prefix}{$key}";
			}
			
			return $this->main->remove($key);
		}
	}
	
	
	
	
	/**
	 * The configuration helper class managing the Configuration->$config
	 * attribute
	 * 
	 * @internal
	 */
	class ConfigurationHelperConfig extends ConfigurationHelper {
		const TYPE_SINGLE     =  0;
		const TYPE_SI_BOOLEAN =  1;
		const TYPE_SI_FLOAT   =  2;
		const TYPE_SI_INTEGER =  4;
		const TYPE_SI_STRING  =  8;
		const TYPE_MULTIPLE   = 64;
		const TYPE_MU_BOOLEAN = 65;
		const TYPE_MU_FLOAT   = 66;
		const TYPE_MU_INTEGER = 68;
		const TYPE_MU_STRING  = 72;
		
		
		
		/**
		 * A list of all parsers to try using with decreasing importance
		 * 
		 * @var array
		 */
		protected $backends;
		
		/**
		 * Configuration item's cache for items which were set before all
		 * backends were loaded
		 * 
		 * @var array
		 */
		protected $cache_preload = array();
		
		/**
		 * Configuration item's cache during runtime (might not be used)
		 * 
		 * @var array
		 */
		protected $cache = array(
		  'system.name'    => array('type' => 4, 'value' => "aCMS"),
		  'system.version' => array('type' => 4, 'value' => "0.1.7"),
		  'system.webpage' => array('type' => 4, 'value' => "http://alexander.nattrans.net/aCMS/"),
		);
		
		/**
		 * Should the internal cache be used? (Not useful for shared config...)
		 * 
		 * @var bool
		 */
		protected $caching = true;
		
		/**
		 * The class initializing this class
		 * 
		 * @var Configuration
		 */
		protected $parent;
		
		/**
		 * Are all back-ends initialized yet?
		 * 
		 * @var bool
		 */
		protected $ready = false;
		
		
		
		/**
		 * Convert a PHP value into a string with the type given
		 * 
		 * @param string  $value
		 *        The PHP value to convert
		 * @param integer $type
		 *        The type to convert from (Default: Auto-detect type)
		 * @return string|null
		 */
		public static function value2string($value, $type) {
			//// Auto-detect type if it is not already given
			if(!is_integer($type)) {
				$type = self::value_find_type($value);
				// Return if auto-detection failed
				if(!is_integer($type)) {
					return null;
				}
			}
			
			// Multi-value types
			if($type == ($type|self::TYPE_MULTIPLE)) {
				// Compose multi-value type of single value types
				$string = '';
				// Iterate over array/iterator items
				foreach((array) $value as $item) {
					// Convert value to string
					$result = self::value2string(
					  $item, $type ^ self::TYPE_MULTIPLE
					);
					// Chain items together
					if(is_string($result)) {
						$string .= str_replace("<", "\\<", $result)."<";
					} else {
						return null;
					}
				}
				return $string;
			} else {
				if($type == self::TYPE_SI_BOOLEAN) {
					if($value) {
						return 't';
					} else {
						return 'f';
					}
				} elseif($type == self::TYPE_SI_FLOAT) {
					return strval(floatval($value));
				} elseif($type == self::TYPE_SI_INTEGER) {
					return strval(intval($value));
				} elseif($type == self::TYPE_SI_STRING) {
					return strval($value);
				} else {
					return null;
				}
			}
		}
		
		
		/**
		 * Convert a string value into a real PHP value
		 * 
		 * @param string  $string
		 *        The string value to convert
		 * @param integer $type
		 *        What to convert the string into
		 * @return mixed The PHP value
		 */
		public static function string2value($strval, $type) {
			if(!is_string($strval)) {
				$strval = "";
			}
			
			// Check for a multiple value type (list)
			if($type == ($type|self::TYPE_MULTIPLE)) {
				$values = array();
				
				$escape = false;
				$chars  = "";
				foreach(str_split($strval) as $char) {
					// Separator (Add content to value list)
					if(!$escape && $char == '<') {
						$values[] = self::string2value(
						  $chars, $type ^ self::TYPE_MULTIPLE
						);
						$chars = "";
					// Begin escape sequence
					} elseif(!$escape && $char == "\\") {
						$escape = true;
					// Character
					} else {
						$chars .= $char;
						$escape = false;
					}
				}
				
				return $values;
			// Check for single type definition
			} elseif($type == ($type|self::TYPE_SINGLE)) {
				switch($type) {
					case self::TYPE_SI_BOOLEAN:
						// Boolean
						if(trim(strtolower($strval)) == "f") {
							return false;
						} elseif(trim(strtolower($strval)) == "t") {
							return true;
						} else {
							return null;
						}
					break;
					case self::TYPE_SI_INTEGER:
						// Integer
						return intval($strval);
					break;
					case self::TYPE_SI_FLOAT:
						// Float
						return floatval($strval);
					break;
					case self::TYPE_SI_STRING:
						// String
						return strval($strval);
					break;
					default:
						return null;
				}
			// Invalid type
			} else {
				return null;
			}
		}
		
		
		/**
		 * Try to find the type of the given value and return it if it could be
		 * detected
		 * 
		 * @param mixed $value
		 *        The PHP value to examine
		 * @return integer|null The type or null if it couldn't be detected
		 */
		public static function value_find_type($value) {
			if(is_array($value)) {
				$type = null;
				foreach($value as $item) {
					// Find type of subvalue
					$type2 = self::value_find_type($item) | self::TYPE_MULTIPLE;
					// Check if the current subvalue type matches last subvalue
					// type (if there was one)
					if(!is_integer($type) || $type == $type2) {
						// Assign new type
						//  (if there was none or if they are identical)
						$type = $type2;
					} else {
						$types = array($type, $type2);
						$types = sort($ypes);
						// Convert booleans to integers if required
						if($types == array(self::TYPE_MU_BOOLEAN, self::TYPE_MU_INTEGER)){
							$type = self::TYPE_MU_INTEGER;
						// Convert booleans to floats if required
						} elseif($types == array(self::TYPE_MU_BOOLEAN, self::TYPE_MU_FLOAT)) {
							$type = self::TYPE_MU_FLOAT;
						// Convert integers to floats if required
						} elseif($types == array(self::TYPE_MU_FLOAT, self::TYPE_MU_INTEGER)) {
							$type = self::TYPE_MU_FLOAT;
						// Non-convertable group, exit
						} else {
							return null;
						}
					}
				}
				
				return $type;
			} else {
				// Boolean
				if(is_bool($value)) {
					return self::TYPE_SI_BOOLEAN;
				// Float
				} elseif(is_float($value)) {
					return self::TYPE_SI_FLOAT;
				// Integer
				} elseif(is_int($value)) {
					return self::TYPE_SI_INTEGER;
				// String (or resource or object)
				} else {
					return self::TYPE_SI_STRING;
				}
			}
		}
		
		
		
		
		/**
		 * @param Configuration $parent
		 *        The class initializing this class
		 * @param array         $data
		 *        The information provided about the different backends and
		 *        settings
		 */
		public function __construct(Configuration $parent, $data=null) {
			$this->parent = $parent;
			
			$parent->config = $this;
			
			//TODO: Implement multiple back-ends and get rid of ConfigurationConnection
				// Create a data class from scope if none given
				if(!is_object($data)) {
					$data = ConfigurationConnection::from_scope();
				}
				
				// Initialize the plugin loader class
				$loader = new PluginLoader($data->type, $parent);
				
				// Load the parser and process data
				$this->backends[] = array(
					'object'   => $loader->load('configuration', $this, $data),
					'readonly' => false
				);
			
			// All backends were loaded it's time to flush the preload-cache
			$this->ready = true;
			// Apply all changes made during preload
			foreach($this->cache_preload as $name => $info) {
				if($this->exists($name)) {
					$this->update($name, $info['value']);
				} else {
					$this->create($name, $info['value'], $info['type']);
				}
			}
			// Clear the preload-cache
			$this->cache_preload = array();
		}
		
		
		/**
		 * Return if this helper was completely initialized
		 * 
		 * @return bool
		 */
		public function is_ready() {
			return $this->ready;
		}
		
		
		/**
		 * Return a configuration item by its name or NULL if it does not exists
		 * 
		 * @param string $name
		 *        The name of the configuration key
		 * @throws ConfigurationKeyNotFoundError
		 * @return array|bool|float|integer|string
		 */
		public function fetch($name) {
			// Skip global prefix
			if(substr($name, 0, 1) == '*') {
				$name = substr($name, 1);
			}
			
			// Return item from cache if possible
			if(isset($this->cache[$name]['value'])) {
				return $this->cache[$name]['value'];
			}
			
			if($this->ready) {
				// Split up configuration key name
				if(strpos($name, '.') !== NULL) {
					list($section, $item) = explode('.', $name, 2);
				} else {
					list($section, $item) = array('', $name);
				}
				
				// Find the first backend implementing this configuration key
				foreach($this->backends as $backend) {
					if(!is_object($backend['object'])) {
						continue;
					}
					
					$data = $backend['object']->item_fetch($section, $item);
					if(!is_null($data)
					&& array_key_exists('value', $data)
					&& array_key_exists('type',  $data)) {
						// Convert value back into a real PHP value
						$value = self::string2value(
						  $data['value'], $data['type']
						);
						
						// Write item into cache
						$this->cache[$name] = array();
						$this->cache[$name]['type'] = $data['type'];
						
						// Only cache item value if item caching is used
						if($this->caching) {
							$this->cache[$name]['value'] = $value;
						}
						
						return $value;
					}
				}
				
				// No backend found - it does not exist!
				throw new ConfigurationKeyNotFoundError('config', $name);
			} else {
				// Let's hope this item is in the preload-cache...
				if(isset($this->cache_preload[$name]['value'])) {
					return $this->cache_preload[$name]['value'];
				} else {
					throw new ConfigurationKeyNotFoundError('config', $name);
				}
			}
		}
		
		
		/**
		 * Return all configuration items in an array where the key is the name
		 * and the value is the configuration item value
		 * 
		 * array('name' => value, ...)
		 * 
		 * @return array
		 */
		public function fetch_all() {
			if($this->ready) {
				$array = array();
				
				// Find the first backend implementing this configuration key
				foreach(array_reverse($this->backends) as $backend) {
					if(!is_object($backend['object'])) {
						continue;
					}
					
					$table = $backend['object']->table_fetch();
					foreach($table as $section => $items) {
						foreach($items as $name => $data) {
							if(!is_null($data)
							&& isset($data['value']) && isset($data['type'])) {
								$key   = "{$section}.{$name}";
								$value = self::string2value(
									$data['value'], $data['type']
								);
								
								// Write item into cache
								$this->cache[$key] = array();
								$this->cache[$key]['type'] = $data['type'];
								// Only cache item value if item caching is used
								if($this->caching) {
									$this->cache[$key]['value'] = $value;
								}
								
								// Write item into array
								$array[$key] = $value;
							}
						}
					}
				}
				
				return $array;
			} else {
				// Return preload cache - it's all we have!
				return $this->cache_preload;
			}
		}
		
		
		/**
		 * Check if a configuration item exists
		 * 
		 * @param string $name
		 *        The configuration key name
		 * @return bool
		 */
		public function exists($name) {
			// Skip global prefix
			if(substr($name, 0, 1) == '*') {
				$name = substr($name, 1);
			}
			
			// Return true if the key exists within the cache
			if(isset($this->cache[$name]['value'])) {
				return true;
			}
			
			if($this->ready) {
				// Split up configuration key name
				if(strpos($name, '.') !== NULL) {
					list($section, $item) = explode('.', $name, 2);
				} else {
					list($section, $item) = array('', $name);
				}
				
				// Return true if any backend implements this key
				foreach($this->backends as $backend) {
					if(is_object($backend['object'])
					&& $backend['object']->item_exists($section, $item)) {
						return true;
					}
				}
				
				// It wasn't found in any backend - Forget it!
				return false;
			} else {
				return isset($this->cache_preload[$name]['value']);
			}
		}
		
		
		/**
		 * Change the value of a configuration item
		 * 
		 * @param string $key
		 *        The name of the configuration item
		 * @param mixed  $value
		 *        The new value of the configuration item
		 * 
		 * @throws ConfigurationKeyNotFoundError
		 */
		public function update($key, $value) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			if($this->ready) {
				// Split up configuration key
				if(strpos($key, '.') !== NULL) {
					list($section, $name) = explode('.', $key, 2);
				} else {
					list($section, $name) = array('', $key);
				}
				
				if(isset($this->cache[$key]['type'])) {
					$type = $this->cache[$key]['type'];
				} else {
					// Ask all backends for the type of this item until we find
					// one that knows this item
					foreach($this->backends as $backend) {
						if(!is_object($backend['object'])) {
							continue;
						}
						
						$data = $backend['object']->item_fetch($section, $name);
						if(!is_null($data)
						&& isset($data['value']) && isset($data['type'])) {
							$type = $data['type'];
							break;
						}
					}
				}
				if(!isset($type)) {
					// No backend knows of this item - return error status
					throw new ConfigurationKeyNotFoundError('config', $name);
				}
				
				$strval = self::value2string($value, $type);
				
				// Update cache item
				$this->cache[$key] = array();
				$this->cache[$key]['type'] = $type;
				// Update cache item value if allowed to
				if($this->caching) {
					$this->cache[$key]['value'] = $value;
				}
				
				// Update item in all writable database
				foreach($this->backends as $backend) {
					if(is_object($backend['object']) && !$backend['readonly']) {
						$backend['object']->item_update($section,$name,$strval);
					}
				}
			} else {
				// Update preload cache item
				if(isset($this->cache_preload[$name]['type'])) {
					$this->cache_preload[$name]['value'] = self::value2string(
					  $value, $this->cache_preload[$name]['type']
					);
					
					// Update successful
					return;
				}
				
				// Item does not exist - failure!
				throw new ConfigurationKeyNotFoundError('config', $name);
			}
		}
		
		
		/**
		 * Create a new configuration item
		 * 
		 * @param string $key
		 *        The name of the configuration item to create
		 * @param mixed  $value
		 *        The initial value of the configuration item
		 * @param string $type
		 *        The type to store in the configuration item
		 */
		public function create($key, $value, $type=null) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			if(!is_integer($type)) {
				$type = self::value_find_type($value);
			}
			
			// Convert value to string
			$strval = self::value2string($value, $type);
			
			if($this->ready) {
				// Split up configuration key
				if(strpos($key, '.') !== NULL) {
					list($section, $name) = explode('.', $key, 2);
				} else {
					list($section, $name) = array('', $key);
				}
				
				// Create cache item
				$this->cache[$key] = array();
				$this->cache[$key]['type'] = $type;
				// Set cache item value if allowed to
				if($this->caching) {
					$this->cache[$key]['value'] = $value;
				}
				
				// Create item in all writable database
				foreach($this->backends as $backend) {
					if(is_object($backend['object']) && !$backend['readonly']) {
						$backend['object']->item_create(
						  $section, $name, $strval, $type
						);
					}
				}
			} else {
				// Write value into preload-cache
				$this->cache_preload[$key] = array(
				  'value' => $value,
				  'type'  => $type
				);
			}
		}
	}
	
	
	
	
	/**
	 * @internal
	 */
	class ConfigurationHelperRuntime extends ConfigurationHelper {
		/**
		 * Configuration runtime cache
		 * 
		 * @var array
		 */
		protected $data = array();
		
		
		
		/**
		 * @param Configuration $parent
		 *        The class initializing this class
		 * @param array         $defaults
		 *        The initial runtime array to use
		 */
		public function __construct(Configuration $parent, array $defaults=array()){
			$this->data = $defaults;
			
			// Set startup time if not already done
			if(!isset($this->data['system.inittime'])) {
				global $INITTIME;
				if(is_float($INITTIME)) {
					$this->data['system.inittime'] = $INITTIME;
				} else {
					$this->data['system.inittime'] = microtime(true);
				}
			}
			// Set system directory if not already done
			if(!isset($this->data['system.dir'])) {
				$this->data['system.dir'] = realpath(__DIR__ . '/' . "..");
			}
		}
		
		
		/**
		 * Return a runtime configuration item
		 * 
		 * @param string $key
		 *        The name of the runtime item
		 * @throws ConfigurationKeyNotFoundError
		 * @return mixed
		 */
		public function fetch($key) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			if(isset($this->data[$key])) {
				return $this->data[$key];
			} else {
				throw new ConfigurationKeyNotFoundError('runtime', $key);
			}
		}
		
		
		/**
		 * Return all runtime configuration items
		 * 
		 * @return array
		 */
		public function fetch_all() {
			return array_slice($this->data, 0);
		}
		
		
		/**
		 * Check if a runtime configuration item exists
		 * 
		 * @param string $key
		 *        The name of the runtime item
		 * @return bool
		 */
		public function exists($key) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			return array_key_exists($key, $this->data);
		}
		
		
		/**
		 * Change the value of a runtime item
		 * 
		 * @param string $key
		 *        The name of the runtime item
		 * @param mixed  $value
		 *        The new value of the cruntime item
		 * @return bool Success?
		 */
		public function update($key, $value) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			$this->data[$key] = $value;
		}
		
		
		/**
		 * Create a new runtime item
		 * 
		 * @param string $key
		 *        The name of the runtime item
		 * @param mixed  $value
		 *        The new value of the runtime item
		 * @return bool Success?
		 */
		public function create($key, $value) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			$this->data[$key] = $value;
		}
		
		
		/**
		 * Remove a runtime item
		 * 
		 * @param string $key
		 *        The name of the runtime item to delete
		 */
		public function remove($key) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			if(isset($this->data[$key])) {
				unset($this->data[$key]);
			} else {
				throw new ConfigurationKeyNotFoundError('runtime', $key);
			}
		}
	}
	
	
	
	/**
	 * @internal
	 */
	class ConfigurationHelperStorage extends ConfigurationHelper {
		/**
		 * Configuration storage cache
		 * 
		 * @var array
		 */
		protected $data = array();
		
		/**
		 * Was this helper initialized yet?
		 * 
		 * @var bool
		 */
		protected $initialized = false;
		
		/**
		 * The parent configuration class
		 * 
		 * @var Configuration
		 */
		protected $parent;
		
		/**
		 * The path of storage file used
		 * 
		 * @var string
		 */
		protected $storagefile;
		
		
		
		/**
		 * @param Configuration $parent
		 *        The configuration main class initializing this class
		 * @param string [$storagefile="%CONFIG:cache.directory%/storage.sdf"]
		 *        The path of the storage file to use
		 */
		public function __construct(Configuration $parent, $storagefile=null) {
			$this->parent      = $parent;
			$this->storagefile = $storagefile;
			$this->data        = array();
			$this->initialized = false;
			
			// Try initalizing with the data provided
			$this->initialize();
			
			// Save storage data to disk on shutdown
			register_shutdown_function(array($this, 'generate'));
		}
		
		
		/**
		 * 
		 */
		protected function initialize() {
			if(!$this->initialized) {
				if(!is_string($this->storagefile)) {
					$config = $this->parent->config();
					
					// Find the storefile path
						
					
					// Check if a cache directory is set
					if(is_object($config)
					&& $config->exists("*cache.directory")
					&& $config->exists("*cache.lifetime")) {
						$this->storagefile = realpath(
								$config->fetch("*cache.directory")
						);
						$this->storagefile.= '/' . "storage.sdf";
						
						if(!file_exists($this->storagefile)) {
							// Storage file does not exist... nothing to parse
							return;
						}
						
						// Calculate how long the cache is still valid
						$age = microtime(true) - filemtime($this->storagefile);
						$lifetime = $config->fetch("*cache.lifetime");
						if($age > $lifetime) {
							// Regenerate data
							$this->data        = array();
							$this->initialized = true;
							return;
						}
					} else {
						// No cache directory set yet... don't parse anything
						return;
					}
				}
				
				$this->parse($this->storagefile);
				$this->initialized = true;
			}
		}
		
		
		/**
		 * Parse the storage file and populate its contents
		 * 
		 * @uses unserialize
		 * 
		 * @param string $storefile The file to parse
		 * @return array
		 */
		protected function parse($storefile) {
			if(!file_exists($storefile)) {
				file_put_contents($storefile, serialize(array()));
			}
			
			// Try reading the storage file
			$data = @unserialize(file_get_contents($storefile));
			if(is_array($data)) {
				$this->data = array_merge($data, $this->data);
			} else {
				file_put_contents($storefile, serialize(array()));
			}
		}
		
		
		/**
		 * Generate a storage file from the current data set and write it to
		 * disk
		 */
		public function generate() {
			if($this->initialized) {
				$content = serialize($this->data);
				file_put_contents($this->storagefile, $content, LOCK_EX);
			}
		}
		
		
		/**
		 * Return a runtime configuration item
		 * 
		 * @param string $key
		 *        The name of the runtime item
		 * @throws ConfigurationKeyNotFoundError
		 * @return mixed
		 */
		public function fetch($key) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			// Initialize this class if possible
			if(!$this->initialized) {
				$this->initialize();
			}
			
			if(isset($this->data[$key])) {
				return $this->data[$key];
			} else {
				throw new ConfigurationKeyNotFoundError('storage', $key);
			}
		}
		
		
		/**
		 * Return all runtime configuration items
		 * 
		 * @return array
		 */
		public function fetch_all() {
			// Initialize this class if possible
			if(!$this->initialized) {
				$this->initialize();
			}
			
			return array_slice($this->data, 0);
		}
		
		
		/**
		 * Check if a runtime configuration item exists
		 * 
		 * @param string $key
		 *        The name of the runtime item
		 * @return bool
		 */
		public function exists($key) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			// Initialize this class if possible
			if(!$this->initialized) {
				$this->initialize();
			}
			
			return isset($this->data[$key]);
		}
		
		
		/**
		 * Change the value of a runtime item
		 * 
		 * @param string $key
		 *        The name of the runtime item
		 * @param mixed  $value
		 *        The new value of the runtime item
		 * @return bool Success?
		 */
		public function update($key, $value) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			// Initialize this class if possible
			if(!$this->initialized) {
				$this->initialize();
			}
			
			$this->data[$key] = $value;
		}
		
		
		/**
		 * Create a new runtime item
		 * 
		 * @param string $name
		 *        The name of the runtime item
		 * @param mixed  $value
		 *        The new value of the runtime item
		 * @return bool Success?
		 */
		public function create($key, $value) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			// Initialize this class if possible
			if(!$this->initialized) {
				$this->initialize();
			}
			
			$this->data[$key] = $value;
		}
		
		
		/**
		 * Remove a runtime item
		 * 
		 * @param string $name
		 *        The name of the runtime item to delete
		 * @throws ConfigurationKeyNotFoundError
		 */
		public function remove($key) {
			// Skip global prefix
			if(substr($key, 0, 1) == '*') {
				$key = substr($key, 1);
			}
			
			// Initialize this class if possible
			if(!$this->initialized) {
				$this->initialize();
			}
			
			if(isset($this->data[$key])) {
				unset($this->data[$key]);
			} else {
				throw new ConfigurationKeyNotFoundError('storage', $key);
			}
		}
	}
	
	
	
	
	/**
	 * Class which stores the configuration and its parsers
	 */
	class Configuration extends AOFObject {
		/**
		 * A config helper object or a proxy using it
		 *
		 * @see ConfigurationHelperConfig
		 * @var ConfigurationHelperConfig|ConfigurationHelperproxy
		 */
		protected $config;
		
		/**
		 * A runtime helper object or a proxy using it
		 *
		 * @see ConfigurationHelperRuntime
		 * @var ConfigurationHelperRuntime|ConfigurationHelperproxy
		 */
		protected $runtime;
		
		/**
		 * A storage helper object or a proxy using it
		 *
		 * @see ConfigurationHelperStorage
		 * @var ConfigurationHelperStorage|ConfigurationHelperproxy
		 */
		protected $storage;
		
		/**
		 * The parents configuration class
		 * 
		 * @var Configuration|null
		 */
		protected $parent;
		
		/**
		 * A string to put in front of all configuration key requests
		 * 
		 * @var string
		 */
		protected $prefix = "";
		
		
		
		/**
		 * @param string              [$prefix='']
		 *        A string to prepend all configuration keys with
		 * @param ConfigurationHelper [$config]
		 *        The config configuration helper object to use
		 * @param ConfigurationHelper [$runtime]
		 *        The runtime configuration helper object to use
		 * @param ConfigurationHelper [$storage]
		 *        The storage configuration helper object to use
		 */
		public function __construct($prefix="",
		  ConfigurationHelper $config  = null,
		  ConfigurationHelper $runtime = null,
		  ConfigurationHelper $storage = null,
		  Configuration       $_parent = null) {
		  	$this->prefix = $prefix;
			$this->parent = $_parent;
			
		  	// * Set or create runtime configuration helper
		  	if(is_object($runtime)) {
		  		$this->runtime = $runtime;
			} else {
		  		$this->runtime = new ConfigurationHelperRuntime($this);
			}
			
			// * Set or create storage configuration helper
			if(is_object($storage)) {
				$this->storage = $storage;
			} else {
				$this->storage = new ConfigurationHelperStorage($this);
			}
			
			// * Set or create config configuration helper
			if(is_object($config)) {
				$this->config = $config;
			} else {
				$this->config = new ConfigurationHelperConfig($this);
			}
			
			// Use proxy to apply prefix
			if(is_string($prefix) && strlen($prefix) > 0) {
				foreach(array('config', 'runtime', 'storage') as $helper) {
					// Replace proxy helper instead of adding another layer
					if($this->{$helper} instanceof ConfigurationHelperproxy) {
						$this->{$helper} = new ConfigurationHelperproxy(
						  $this->{$helper}->get_main(),
						  $this->{$helper}->get_prefix() . $prefix
						);
					} else {
						$this->{$helper} = new ConfigurationHelperproxy(
						  $this->{$helper}, $prefix
						);
					}
				}
			}
			
			// Temporary hack
			//TODO: Undo core plugin hack
			if(!defined("_LOADED_CORE_PLUGIN")) {
				if($this->config instanceof ConfigurationHelperproxy) {
					$ready = $this->config->get_main()->is_ready();
				} else {
					$ready = $this->config->is_ready();
				}
				
				if($ready) {
					define("_LOADED_CORE_PLUGIN", 1);
					
					PluginLoader::load_plugin("core", "core", $this->main());
				}
			}
		}
		
		
		/**
		 * Return the config helper object used
		 * 
		 * <u>This is the recommended way of accessing the config data</u>
		 * 
		 * @see ConfigurationHelperConfig
		 * @return ConfigurationHelperConfig|ConfigurationHelperproxy
		 */
		public function config() {
			return $this->config;
		}
		
		
		/**
		 * Return the runtime helper object used
		 * 
		 * <u>This is the recommended way of accessing the runtime data</u>
		 * 
		 * @see ConfigurationHelperRuntime
		 * @return ConfigurationHelperRuntime|ConfigurationHelperproxy
		 */
		public function runtime() {
			return $this->runtime;
		}
		
		
		/**
		 * Return the storage helper object used
		 * 
		 * <u>This is the recommended way of accessing the storage data</u>
		 * 
		 * @see ConfigurationHelperStorage
		 * @return ConfigurationHelperStorage|ConfigurationHelperproxy
		 */
		public function storage() {
			return $this->storage;
		}
		
		
		/**
		 * Create a new instance of the configuration class
		 * 
		 * It will be using the same helpers as the current instance unless
		 * others are provided. This is useful if you want to (for instance set
		 * a prefix or replace one of the helpers classes used).
		 * 
		 * @param string              [$prefix='']
		 *        A string to prepend all configuration keys with
		 * @param ConfigurationHelper [$config]
		 *        The config configuration helper object to use
		 * @param ConfigurationHelper [$runtime]
		 *        The runtime configuration helper object to use
		 * @param ConfigurationHelper [$storage]
		 *        The storage configuration helper object to use
		 * 
		 * @return Configuration
		 */
		public function create_new($prefix=null,
		  ConfigurationHelper $config  = null,
		  ConfigurationHelper $runtime = null,
		  ConfigurationHelper $storage = null) {
			$prefix  = is_string($prefix)  ? $prefix  : $this->prefix;
			$config  = is_object($config)  ? $config  : $this->config;
			$runtime = is_object($runtime) ? $runtime : $this->runtime;
			$storage = is_object($storage) ? $storage : $this->storage;
			
			return new Configuration(
			  $prefix, $config, $runtime, $storage, $this
			);
		}
		
		
		/**
		 * Get the parent of this Configuration instance or null if there isn't
		 * any
		 * 
		 * @return Configuration|null
		 */
		public function parent() {
			return $this->parent;
		}
		
		
		/**
		 * Get the top-level parent of this Configuration instance or this one
		 * if there isn't any higher instance
		 * 
		 * @uses parent
		 * 
		 * @return Configuration
		 */
		public function main() {
			$current = $this;
			while($current->parent() !== NULL) {
				$current = $current->parent();
			}
			
			return $current;
		}
		
		
		/**
		 * Return all configuration items
		 * 
		 * @return array
		 */
		public function get_config_array() {
			return $this->config->fetch_all();
		}
		
		/**
		 * Return a configuration item by its name or `VALUE_INVALID` if the
		 * item does not exist
		 * 
		 * @param string $name The name of the configuration item
		 * @return mixed
		 */
		public function get_config($name) {
			return $this->config->fetch($name);
		}
		
		/**
		 * Check if a configuration item exists based on the name given
		 * 
		 * @param string $name The name of the configuration item
		 * @return bool
		 */
		public function has_config($name) {
			return $this->config->exists($name);
		}
		
		/**
		 * Create a new configuration item
		 * 
		 * @param string  $name  The name of the configuration item
		 * @param mixed   $value The new value of the configuration item
		 * @param integer $type  The type of the new configuration item
		 * @return bool Success?
		 */
		public function new_config($name, $value, $type=null) {
			return $this->config->create($name, $value, $type);
		}
		
		/**
		 * Change the value of a configuration item
		 * 
		 * @param string $name  The name of the configuration item
		 * @param mixed  $value The new value of the configuration item
		 * @return bool Success?
		 */
		public function set_config($name, $value) {
			return $this->config->update($name, $value);
		}
		
		
		
		/**
		 * Fetch an item from the run-time array
		 * 
		 * @param string $item The name of the item
		 * @return mixed
		 */
		public function get_runtime($item) {
			if($this->runtime->exists($item)) {
				return $this->runtime->fetch($item);
			} else {
				return null;
			}
		}
		
		/**
		 * Fetch all items from the run-time array
		 * 
		 * @return array
		 */
		public function get_runtime_array() {
			return $this->runtime->fetch_all();
		}
		
		/**
		 * Check if an item is part of the run-time array
		 * 
		 * @param string $item The item to check
		 * @return bool
		 */
		public function has_runtime($item) {
			return $this->runtime->exists($item);
		}
		
		/**
		 * Change an item from the run-time array
		 * 
		 * @param string $item  The name of the item
		 * @param mixed  $value The new value of the item
		 */
		public function set_runtime($item, $value) {
			$this->runtime->update($item, $value);
		}
		
		
		
		/**
		 * Fetch all items from the storage array
		 * 
		 * @return array
		 */
		public function get_storage_array() {
			return $this->storage->fetch_all();
		}
		
		/**
		 * Fetch an item from the storage array
		 * 
		 * @param string $item The name of the item
		 * @return mixed
		 */
		public function get_storage($item) {
			return $this->storage->fetch($item);
		}
		
		/**
		 * Check if an item is part of the storage array
		 * 
		 * @param string $item The item to check
		 * @return bool
		 */
		public function has_storage($item) {
			return $this->storage->exists($item);
		}
		
		/**
		 * Change an item from the storage array
		 * 
		 * @param string $item  The name of the item
		 * @param mixed  $value The new value of the item
		 */
		public function set_storage($item, $value) {
			return $this->storage->update($item, $value);
		}
	}
