<?php
	/**
	 * The class compiles a given string into an executable PHP script which
	 * can be loaded instead of a requested resource
	 */
	class RequestOutputStreamCache extends AOFObject {
		/**
		 * @var Configuration
		 */
		protected $config;
		
		
		
		/**
		 * Process the content given and compile it into native PHP code
		 * 
		 * @param string $content
		 *        The content which should be compiled
		 * 
		 * @return string
		 */
		static public function compile($content) {
			// "Escape" all PHP start tags
			$content = str_replace("<?php ", "<?php print('<?php '); ?>", $content);
			
			// "Escape" all ASP start tags (PHP ini asp_tags)
			$content = str_replace("<%", "<?php print('<%'); ?>", $content);
			
			// Match all meta commands and process them
			$blocks = array();
			preg_match_all(
				'/<\\$(\w+)((?:\s+(?:["][^"]*["]|[\'][^\']*[\']|\w+)\s*)*)\s*\\$>/i',
				$content,
				$blocks,
				PREG_SET_ORDER
			);
			$plugin_count = 0;
			foreach($blocks as $block) {
				// Determine the command which was called
				$command = strtoupper($block[1]);
				
				// Determine the arguments used for the command called
				preg_match_all(
					'/\s*(["][^"]*["]|[\'][^\']*[\']|\w+)\s*/i',
					$block[2],
					$entries,
					PREG_PATTERN_ORDER
				);
				$arguments = array();
				foreach($entries[1] as $entry) {
					// Remove quotes from each argument and store them
					if((substr($entry, 0, 1) == '"'
					 || substr($entry, 0, 1) == "'")
					&& (substr($entry, -1) == '"'
					 || substr($entry, -1) == "'")) {
						$arguments[] = substr($entry, 1, -1);
					} else {
						$arguments[] = $entry;
					}
				}
				
				switch($command) {
					case 'LT': // Escaped sequence
						$replace = '<';
					break;
					case 'PROCESS_PLUGIN': // Load output plugin
						$replace = self::compile_plugin($arguments);
						$plugin_count++;
					break;
					case 'HEADER':
						$replace = "<?php ";
						foreach($arguments as $header) {
							$header = str_replace("'", "\\'", $header);
							
							$replace.= "header('{$header}');";
						}
						$replace.= " ?>";
					break;
					default:
						report2(WARNING,
							"Invalid processing command: {$command}"
						);
						
						$replace = "";
				}
				
				$content = str_replace($block[0], $replace, $content);
			}
			
			return array($content, $plugin_count);
		}
		
		
		static public function compile_plugin($arguments) {
			$content  = "";
			$content .= "<?php print(PluginLoader::load_plugin(";
			$content .= "'{$arguments[0]}', '{$arguments[1]}', ";
			$content .= "\$CONFIGURATION, null)";
			$content .= "->process('{$arguments[2]}', ";
			$content .= "{$arguments[3]})); ?>";
			
			return $content;
		}
		
		
		
		/**
		 * @param string $stream
		 *        The name of the stream who's resource should be loaded
		 * @param string $resource 
		 *        The name of the resource which should be loaded
		 */
		public function __construct($stream, $identifier) {
			$this->stream   = $stream;
			$this->resource = $identifier;
			
			global $CONFIGURATION;
			$this->config = $CONFIGURATION->create_new('cache.');
			
			$this->cache_name = "{$this->stream}.{$this->resource}";
			
			// Read the contents of the storaged cache entry
			if($this->config->storage()->exists($this->cache_name)) {
				$this->cache=$this->config->storage()->fetch($this->cache_name);
			} else {
				$this->cache=array();
			}
		}
		
		
		/**
		 * Run the contents of the cache file
		 * 
		 * 
		 */
		public function cache_execute() {
			$filepath = $this->cache_find_filepath();
			
			// Check if the client supports gzip compression
			$accepts_gzip = false;
			if($this->config->config()->fetch("compress")) {
				foreach(explode(',',$_SERVER['HTTP_ACCEPT_ENCODING']) as $encoding){
					if(strtolower(trim($encoding)) == 'gzip') {
						$accepts_gzip = true;
						break;
					}
				}
				unset($encoding);
			}
			
			// Try to optimizing requests to static resources
			if(!$this->cache['dynamic'] && !headers_sent()) {
				// $_SERVER['HTTP_IF_MODIFIED_SINCE'] == 'Sun, 25 Dec 2011 00:08:50 GMT'
				// $_SERVER['HTTP_IF_NONE_MATCH'] == '-1503056294'
				
				// Send expiry header containing the remaining cache lifetime
				header('Expires: ' . gmdate(
					'D, d M Y H:i:s T',
					microtime(true) + $this->cache_get_remaining()
				));
				
				// Send Last-Modified-Headers based on when the file was
				// modified last
				header('Last-Modified: ' . gmdate(
					'D, d M Y H:i:s T',
					filemtime($filepath)
				));
				
				// Process If-Modifed-Since headers sent by the client
				if(isset($_SERVER['HTTP_IF_MODIFIED_SINCE'])) {
					// Parse the last modification time
					$modified = strtotime(preg_replace(
						'/;.*$/', '', $_SERVER["HTTP_IF_MODIFIED_SINCE"]
					));
					
					// Check if the cache file was modified since
					if($modified >= filemtime($filepath)) {
						
						
						// No modifications
						header("HTTP/1.0 304 Not Modified");
						return;
					}
				}
				
				// Caculate the checksum of the contents sent to the client
				$checksum = crc32(file_get_contents($filepath)).'';
				
				// Send ETag with a hash of the compressed content
				header('ETag: ' . $checksum);
				
				// Process If-None-Match headers sent by the client
				if(isset($_SERVER['HTTP_IF_NONE_MATCH'])) {
					// Check if the cache file was modified since
					if($checksum == $_SERVER['HTTP_IF_NONE_MATCH']) {
						// No modifications
						header("HTTP/1.0 304 Not Modified");
						return;
					}
				}
			}
			
			if($accepts_gzip && !headers_sent()) {
				// Clean environment
				unset($accepts_gzip);
				
				// Send gzip headers
				header('Content-Encoding: gzip');
				header('Vary: Accept-Encoding');
				
				if(file_exists("{$filepath}.gz")) {
					$content = file_get_contents("{$filepath}.gz");
					
					// Send ETag with a hash of the compressed content
					// (crc32 is not the best hashing algorithm but it is the
					// fastest that php has to offer)
					$checksum = crc32($content);
					header('ETag', $checksum);
					
					// Process If-None-Match headers sent by the client
					if(isset($_SERVER['HTTP_IF_NONE_MATCH'])) {
						// Check if the cache file was modified since
						if($checksum == $_SERVER['HTTP_IF_NONE_MATCH']) {
							// No modifications
							header("HTTP/1.0 304 Not Modified");
							return;
						}
					}
					
					// Send pre-compressed output
					print($content);
				} else {
					// Compress output before sending it to the client
					ob_start('gzencode');
					$this->include_file($filepath);
					ob_end_flush();
				}
			} else {
				$this->include_file($filepath);
			}
		}
		
		
		/**
		 * Include a single file
		 * 
		 * Using this function ensures a clean environment for the included file
		 * 
		 * @param string $filepath
		 *        The filepath of the file to include
		 * 
		 * @return integer
		 */
		public function include_file($filepath) {
			// Provide configuration object as global
			global $CONFIGURATION;
			
			return (include($filepath));
		}
		
		
		/**
		 * Return if the resource requested is cached
		 * 
		 * @return bool
		 */
		public function cache_exists() {
			return file_exists($this->cache_find_filepath());
		}
		
		
		/**
		 * Return the filepath in which the cache of this resource should be
		 * stored
		 * 
		 * @return string
		 */
		public function cache_find_filepath() {
			$directory = "{$this->config->config()->fetch('directory')}/";
			$directory.= "content" . '/' . urlencode($this->stream);
			
			// Create cache directory if it does not already exist
			if(!is_dir($directory)) {
				mkdir($directory, 0777, true);
			}
			
			// Return the file within the specific caching directory
			return $directory . '/' . urlencode($this->stream) .
					':' . urlencode($this->resource);
		}
		
		
		/**
		 * Get the remaining time left for the cached file
		 * 
		 * This value may be negative if we are past the maximum cache time.
		 * 
		 * @return float
		 */
		public function cache_get_remaining() {
			// Find the cache lifetime used
			if(isset($this->cache['lifetime'])
			&& is_integer($this->cache['lifetime'])) {
				$lifetime = $this->cache['lifetime'];
			} else {
				$lifetime = $this->config->config()->fetch('lifetime');
			}
			
			// Find the age of cache file
			if(file_exists($this->cache_find_filepath())) {
				$age = filemtime($this->cache_find_filepath());
			} else {
				$age = 0;
			}
			
			// Calculate how long the cache is still valid
			return -microtime(true) + $age + $lifetime;
		}
		
		
		/**
		 * Return if the resource requested exists and should be still be used
		 * for caching
		 * 
		 * @return bool
		 */
		public function cache_is_valid() {
			// Check if the cache file exists
			if(!$this->cache_exists()) {
				return false;
			}
			
			// Check if the cache entry is valid
			if(!isset($this->cache['generated'])
			|| !is_float($this->cache['generated'])
			|| !isset($this->cache['dynamic'])
			|| !is_bool($this->cache['dynamic'])){
				return false;
			}
			
			// Check if the requested resources' cache lifetime has expired
			if($this->cache_get_remaining() < 1) {
				return false;
			}
			
			// Return true if all checks were ok
			return true;
		}
		
		
		/**
		 * Store the compiled data given in the cache file
		 * 
		 * @param string $compiled
		 *        The compiled cache data given
		 * @param bool   $is_dynamic
		 *        Are there any dynamic contents within the compiled cache
		 *        document?
		 */
		public function cache_put($compiled, $is_dynamic) {
			// Update its generation time
			$this->cache['generated'] = microtime(true);
			// Update if the content whether is dynamict
			$this->cache['dynamic'] = $is_dynamic;
			// Update the cache data value
			$this->config->storage()->update($this->cache_name, $this->cache);
			
			$filepath = $this->cache_find_filepath();
			
			// Write compiled data into the cache file
			file_put_contents($filepath, $compiled);
			
			// Pre-generate gzipped content if the content isn't dynamic
			if(!$is_dynamic && $this->config->config()->fetch("precompress")) {
				file_put_contents(
					"{$filepath}.gz", gzencode($compiled, 9)
				);
			}
			
			// Find the cache lifetime used
			if(isset($this->cache['lifetime'])
			&& is_integer($this->cache['lifetime'])) {
				$lifetime = $this->cache['lifetime'];
			} else {
				$lifetime = $this->config->config()->fetch('lifetime');
			}
		}
		
		
		/**
		 * Return the contents of the cache file
		 * 
		 * @return string
		 */
		public function cache_read() {
			return file_get_contents($this->cache_find_filepath());
		}
		
		
		/**
		 * Compile and cache the content given
		 *
		 * @param string $content
		 *        The content which should be processed
		 */
		public function process_content($content) {
			list($compiled, $plugins_loaded) = self::compile($content);
			$this->cache_put($compiled, (bool)$plugins_loaded);
		}
	}
