<?php
	/**
	 * @package aCMS-AOF
	 */
	require_once("AOFError.class.php");
	
	
	
	/**
	 * Base exception class for all errors in an AOFObject
	 */
	class AOFObjectError extends AOFError {
		/**
		 * The title of the error
		 * @var string
		 */
		protected $title = 'API error: $CLASS';
		
		
		/**
		 * The object which has thrown the error
		 * @var mixed
		 */
		public $object;
		
		/**
		 * The name of the class which has thrown the error
		 * @var string
		 */
		public $class = '';
		
		/**
		 * The name of the attribute on which something failed
		 * @var string
		 */
		public $name = '';
		
		/**
		 * A list of arguments supplied
		 * @var array
		 */
		public $arguments = array();
		
		
		
		/**
		 * @param mixed  $object    The object failing to load the method
		 * @param string $funcname  The name of the method
		 * @param array  $arguments A list of arguments supplied
		 */
		public function __construct($object, $name, $arguments) {
			$this->object    = $object;
			$this->class     = get_class($this->object);
			$this->name      = $name;
			$this->arguments = $arguments;
		}
	}
	
	
	/**
	 * Base exception class for all method-related errors in an AOFObject
	 */
	class AOFObjectMethodError extends AOFObjectError {
		/**
		 * @param mixed  $object    The object failing to load the method
		 * @param string $funcname  The name of the method
		 * @param array  $arguments A list of arguments supplied
		 */
		public function __construct($object, $name, $arguments, $msg) {
			parent::__construct($object, $name, $arguments);
			
			$this->message = "The method {$this->class}::{$name} {$msg}";
		}
	}
	
	
	/**
	 * Exception class if a called method is not defined in an AOFObject
	 */
	class AOFObjectMethodNotDefinedError extends AOFObjectMethodError {
		/**
		 * @param mixed  $object    The object failing to load the method
		 * @param string $funcname  The name of the method
		 * @param array  $arguments A list of arguments supplied
		 */
		public function __construct($object, $name, $arguments) {
			parent::__construct($object, $name, $arguments,
			  "does not exist");
		}
	}
	
	
	/**
	 * Exception class if a method was not called with enough arguments
	 */
	class AOFObjectMethodArgumentCountError extends AOFObjectMethodError {
		/**
		 * @param mixed   $object    The object
		 * @param string  $funcname  The name of the method
		 * @param array   $arguments A list of arguments supplied
		 * @param integer $count     The number of arguments expected
		 */
		public function __construct($object, $name, $arguments, $expected) {
			parent::__construct($object, $name, $arguments,
			  "was not called with enough arguments. Expected: {$expected}");
		}
	}
	
	
	/**
	 * Base exception class for all property-related errors in an AOFObject
	 */
	class AOFObjectPropertyError extends AOFObjectError {
		/**
		 * @param mixed  $object The object failing to load the method
		 * @param string $name   The name of the property
		 */
		public function __construct($object, $name, $msg) {
			parent::__construct($object, $name, array());
			
			$this->message = "The property {$this->class}::\${$name} {$msg}";
		}
	}
	
	
	/**
	 * Exception class for a property which is not defined in an AOFObject
	 */
	class AOFObjectPropertyNotDefinedError extends AOFObjectPropertyError {
		/**
		 * @param mixed  $object The object failing to load the method
		 * @param string $name   The name of the property
		 */
		public function __construct($object, $name) {
			parent::__construct($object, $name, "does not exist");
		}
	}
	
	
	
	
	/**
	 * Abstract base class for all objects
	 */
	abstract class AOFObject {
		/**
		 * A list of secondary parent objects to use
		 * 
		 * @var list
		 */
		protected $parents = array();
		
		/**
		 * An array of initialized parent objects
		 * 
		 * @internal
		 * @var array
		 */
		private $__AOFObject_parents = array();
		
		
		
		
		/**
		 * Generate the secondary parent objects
		 */
		private function __AOFObject_generate_parents() {
			foreach($this->parents as $classname) {
				$parentname = "__AOFObject_parent_{$classname}";
				// Generate customized class (UUUGGGLLLYYY!!!)
				if(!class_exists($parentname)) {
					eval('
						class '.$parentname.' extends '.$classname.' {
							// Skip initialization
							public function __construct() {}
							
							// Call internal function
							public function __AOFObject_func_call($name, $args){
								return call_user_func_array(
									array($this, $name), $args);
							}
							
							// Get internal variable
							public function __AOFObject_prop_get($name) {
								return $this->{$name};
							}
							
							// Does a internal variable exist?
							public function __AOFObject_prop_isset($name) {
								return isset($this->{$name});
							}
							
							// Set internal variable
							public function __AOFObject_prop_set($name, $value){
								$this->{$name} = $value;
							}
							
							// Unset internal variable
							public function __AOFObject_prop_unset($name) {
								if(isset($this->{$name})) {
									unset($this->{$name});
									return true;
								} else {
									return false;
								}
							}
						}
					');
				}
				
				// Initialize new class
				if(!isset($this->__AOFObject_parents[$classname])) {
					$this->__AOFObject_parents[$classname] = new $parentname;
				}
			}
		}
		
		
		/**
		 * Overloading function for all function calls
		 * 
		 * This handles all invalid function calls
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.method
		 * 
		 * @param string $name      The name of the function to emulate
		 * @param array  $arguments A list of arguments for these functions
		 * 
		 * @throws AOFObjectMethodNotDefinedError
		 * 
		 * @return mixed|null
		 */
		public function __call($name, $arguments) {
			if(!$this->__AOFObject_parents) {
				$this->__AOFObject_generate_parents();
			}
			
			foreach($this->__AOFObject_parents as $parent) {
				try {
					// Check if the method exists
					new ReflectionMethod($parent, $name);
					
					return $parent->__AOFObject_func_call($name, $arguments);
				} catch (ReflectionMethod $e) {}
			}
			
			throw new AOFObjectMethodNotDefinedError($this, $name, $arguments);
		}
		
		
		/**
		 * Overloading function for all properties
		 * 
		 * Handles all invalidly fetched variables.
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.members
		 * 
		 * @param string $name The name of the data variable to return
		 * 
		 * @throws AOFObjectPropertyNotDefinedError
		 * 
		 * @return mixed|null
		 */
		public function __get($name) {
			if(!$this->__AOFObject_parents) {
				$this->__AOFObject_generate_parents();
			}
			
			// Check if one of the secondary parent objects has this property
			foreach($this->__AOFObject_parents as $parent) {
				if($parent->__AOFObject_prop_isset($name)) {
					return $parent->__AOFObject_prop_get($name);
				}
			}
			
			throw new AOFObjectPropertyNotDefinedError($this, $name);
		}
		
		
		/**
		 * Overloading function to check if a non-standard property exists
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.members
		 * 
		 * @param string $name The name of the variable to check for
		 * 
		 * @return bool
		 */
		public function __isset($name) {
			if(!$this->__AOFObject_parents) {
				$this->__AOFObject_generate_parents();
			}
			
			// Check if one of the secondary parents has this property
			foreach($this->__AOFObject_parents as $parent) {
				if($parent->__AOFObject_prop_isset($name)) {
					return true;
				}
			}
			
			return false;
		}
		
		
		/**
		 * Overloading function to change the value of a non-standard variable
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.members
		 * 
		 * @param string $name  The name of the data variable to set
		 * @param mixed  $value The value to change the data variable to
		 */
		public function __set($name, $value
		//, $_parents_only=false
		) {
			if(!$this->__AOFObject_parents) {
				$this->__AOFObject_generate_parents();
			}
			
			// Change the property in the current object if requested
			//if(!$_parents_only) {
				$this->{$name} = $value;
			//}
			
			// Set this property for all secondary parents
			foreach($this->__AOFObject_parents as $parent) {
				$parent->__AOFObject_prop_set($name, $value);
			}
		}
		
		
		/**
		 * Overloading function to remove a non-standard variable
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.members
		 * 
		 * @param string $name The name of the data variable to unset
		 * 
		 * @throws AOFObjectPropertyNotDefinedError
		 * 
		 * @return bool Success?
		 */
		public function __unset($name
		//, $_parents_only=false
		) {
			if(!$this->__AOFObject_parents) {
				$this->__AOFObject_generate_parents();
			}
			
			$unset = false;
			
			// Delete the variable in the current object if requested
			//if(!$_parents_only) {
				if(isset($this->{$name})) {
					unset($this->{$name});
					// Show that there was really a value unset
					$unset = true;
				}
			//}
			
			// Unset this property for all secondary parents
			foreach($this->__AOFObject_parents as $parent) {
				if($parent->__AOFObject_prop_unset($name)) {
					$unset = true;
				}
			}
			
			// Throw an exception if no parent has this property
			if(!$unset) {
				throw new AOFObjectPropertyNotDefinedError($this, $name);
				return false;
			} else {
				return true;
			}
		}
	}
	
	
//	class AOFObjectTestParent {
//		protected $test = 'TEST';
//		
//		protected function test() {
//			return 'TEST';
//		}
//	}
//	
//	class AOFObjectTest extends AOFObject {
//		protected $parents = array('AOFObjectTestParent');
//		
//		public function parent_prop_test() {
//			if($this->test == 'TEST') {
//				print('SUCCESS!');
//				return true;
//			} else {
//				print('FAILED!');
//				return false;
//			}
//		}
//		
//		public function parent_func_test() {
//			if($this->test() == 'TEST') {
//				print('SUCCESS!');
//				return true;
//			} else {
//				print('FAILED!');
//				return false;
//			}
//		}
//	}
//	
//	$test = new AOFObjectTest;
//	$test->parent_prop_test();
//	$test->parent_func_test();
