<?php
	/**
	 * @package aCMS-AOF
	 */
	/**
	 * Base class for all AOF errors
	 */
	class AOFError extends Exception {
		/**
		 * The message of the error
		 * 
		 * @var string
		 */
		protected $message;
		
		/**
		 * Additinal error information
		 * 
		 * @var string
		 */
		protected $infos = '';
		
		/**
		 * A list of headers
		 *
		 * @var array
		 */
		protected $headers = array('HTTP/1.0 500 Internal Server Error');
		
		/**
		 * An array of possible solutions
		 * 
		 * @var array
		 */
		protected $solutions = array();
		
		/**
		 * The CSS-stylesheet definitions of the error page
		 * 
		 * @var string
		 */
		protected $stylesheet = '
.trace .frame {
	font-family: Monospace;
	font-size:   1.2em;
}

.trace .frame .class {
	color: #55BB55;
}

.trace .frame .func {
	color: #008800;
}

.trace .frame .sep {
	color: #FF0000;
}

.trace .frame .string {
	color: #FF00FF;
	font-weight: bold;
}

.trace .frame .bool, .trace .frame .array {
	color: #999900;
	font-style: italic;
}';
		
		/**
		 * The page template of the error page
		 * 
		 * The following pseudo-variables will be substituted:
		 *  + $BODY:  The body of the page (see `$template_body`)
		 *  + $CLASS: The current class name
		 *  + $TITLE: The title of the page
		 * 
		 * @var string
		 */
		protected $template = '
<!DOCTYPE html>
<html>
	<head>
		<title>$TITLE</title>
		<style type="text/css">
$STYLESHEET
		</style>
	</head>
	<body>
$BODY
	</body>
</html>';
		
		/**
		 * The body template of the error page
		 * 
		 * The following pseudo-variables will be substituted:
		 *  + $CLASS:     The current class name
		 *  + $MESSAGE:   The error message
		 *  + $SOLUTIONS: The generated list of possible solutions
		 *  + $TITLE:     The title of the page
		 */
		protected $template_body = '
		<h1>$TITLE</h1>
		<p>$MESSAGE</p>
$INFO
$SOLUTIONS
$TRACE';
		
		/**
		 * The title of the error page
		 * 
		 * The following pseudo-variables will be substituted:
		 *  + $CLASS: The current class name
		 * 
		 * @var string
		 */
		protected $title = 'A fatal error occurred: $CLASS';
		
		/**
		 * The error template of the class
		 * 
		 * @var class
		 */
		protected $tpl = '';
		
		/**
		 * An error template instance
		 * 
		 * @var mixed
		 */
		private $_tpl;
		
		
		
		/**
		 * @param array  $bt
		 *        A `debug_backtrace`-comaptible backtrace array
		 * @param bool  [$find_manual_urls=true]
		 *        Show manual URLs we looked up for the entries
		 *        (Performance bumper!)
		 */
		static public function create_trace_html($bt, $find_manual_urls=true) {
			$trace = "<ol class=\"trace\">\n";
			foreach($bt as $frame) {
				$trace .= "\t<li class=\"frame\">";
				
				if($find_manual_urls) {
					if(isset($frame['class'])
					&&($url=self::get_manual_url($frame['class'], 'class'))) {
						$frame['class'] = "<a href=\"$url\">{$frame['class']}</a>";
					}

					if(isset($frame['function'])
					&&($url=self::get_manual_url($frame['function'], 'function'))){
						$frame['function'] = "<a href=\"$url\">{$frame['function']}</a>";
					}
				}
				
				// Generate filepath and line number information
				if(isset($frame['file']) && isset($frame['line'])) {
					$trace.= "File ";
					$trace.= "<span class=\"sep\">&quot;</span>";
					$trace.= "<span class=\"file\">{$frame['file']}</span>";
					$trace.= "<span class=\"sep\">&quot;</span>";
					$trace.= " on Line ";
					$trace.= "<span class=\"line\">{$frame['line']}</span> ";
					$trace.= "<br />";
				}
				
				// Generate the function name
				if(isset($frame['class']) && isset($frame['type'])) {
					$trace.= "<span class=\"class\">{$frame['class']}</span>";
					$trace.= "<span class=\"sep\">{$frame['type']}</span>";
					$trace.= "<span class=\"func\">{$frame['function']}</span>";
				} else {
					$trace.= "<span class=\"func\">{$frame['function']}</span>";
				}
				
				// Generate the argument list
				if(isset($frame['args'])) {
					$trace .= "<span class=\"sep bracket\">(</span>";

					foreach($frame['args'] as $idx => $arg) {
						if($idx > 0) {
							$trace .= '<span class="sep coma">,</span> ';
						}

						if(is_array($arg)) {
							$trace .= '<span class="array">Array</span>';
							$trace .= '<span class="sep cbracket">{</span>';
							$trace .= '<span class="number">'.count($arg).'</span>';
							$trace .= '<span class="sep cbracket">}</span>';
						} elseif(is_bool($arg) && $arg == true) {
							$trace .= '<span class="bool true">true</span>';
						} elseif(is_bool($arg) && $arg == false) {
							$trace .= '<span class="bool false">false</span>';
						} elseif(is_float($arg) || is_int($arg)) {
							$trace .= '<span class="number">'.$arg.'</span>';
						} elseif(is_null($arg)) {
							$trace .= '<span class="null">null</span>';
						} elseif(is_object($arg)) {
							$trace .= '<span class="sep abracket">&lt;</span>';
							$trace .= '<span class="class">'.get_class($arg).'</span>';
							$trace .= '<span class="sep abracket">&gt;</span>';
						} else {
							$trace .= '<span class="sep quote">&quot;</span>';
							$trace .= '<span class="string">'.$arg.'</span>';
							$trace .= '<span class="sep quote">&quot;</span>';
						}
					}
					$trace .= "<span class=\"sep bracket\">)</span>";
				}
				
				$trace .= "</li>\n";
			}
			$trace .= "</ol>\n";
			return $trace;
		}
		
		
		static public function get_manual_url($topic, $type='') {
			static $refdata;
			
			$topic = strtolower($topic);
			$type  = strtolower($type);
			
			$reffile = dirname(__FILE__) . '/' . "phpmanualref.json";
			
			// Don't do anything if the manual cache file cannot be read
			if(!is_readable($reffile)) {
				return false;
			}
			
			// Read manual data if not already done
			if(!is_array($refdata)) {
				$refdata = json_decode(file_get_contents($reffile), true);
			}
			
			// Try to find the URL in the file
			foreach($refdata as $ref) {
				// Validate the reference
				if(!isset($ref['topic'])
				|| !isset($ref['type'])
				|| !isset($ref['url'])) {
					continue;
				}
				
				// Check if the reference name equals our manual topic
				if($topic != $ref['topic']) {
					continue;
				}
				
				// Check if the type is correct
				if($type && $type != $ref['type']) {
					continue;
				}
				
				return $ref['url'];
			}
			
			return false;
		}
		/*static public function get_manual_url($topic) {
			#static $test = false;
			#if($test) {return false;} else {$test=true;}
			
			static $mirror;
			
			// Create stream context
			if(version_compare(PHP_VERSION, '5.2.10', '<')) {
				// PHP below the version 5.2.10 does not support the
				// `ignore_errors` context data element so we have to allow PHP
				// to follow the redirect
				$context = stream_context_create(array(
					'http' => array(
						'method'          => 'HEAD',
						'timeout'         => 1
					)
				));
			} else {
				// Don't follow redirects
				$context = stream_context_create(array(
					'http' => array(
						'method'          => 'HEAD',
						'max_redirects'   => 0,
						'timeout'         => 1,
						'ignore_errors'   => true
					)
				));
			}
			
			if(!$mirror) {
				// Send an invalid manual lookup request to the PHP main server
				// and check were we are redirected too
				$request = fopen(
					"http://php.net/manual-lookup.php", 'r', false, $context);
				
				// Grab the metadata of the request
				$metadata = stream_get_meta_data($request);
				fclose($request);
				
				// Look in the list of response header fields for the
				// "Location:"-header and use its hostname as mirror server
				foreach($metadata['wrapper_data'] as $header) {
					if(strpos($header, ':') !== NULL) {
						list($name, $value) = explode(':', $header, 2);
						if(strtolower(trim($name)) == 'location') {
							$mirror = parse_url(trim($value), PHP_URL_HOST);
							break;
						}
					}
				}
				
				unset($metadata, $header, $name, $value);
			}
			
			// Send the real manual lookup request to the PHP mirror server
			$lookup = fopen(
				"http://{$mirror}/manual-lookup.php?pattern={$topic}",
				'r', false, $context);
			
			// Grab the metadata of the request
			$metadata = stream_get_meta_data($lookup);
			fclose($lookup);
			
			// Look in the list of response header fields for the
			// "Location:"-header and use it as manual URL
			foreach($metadata['wrapper_data'] as $header) {
				if(strpos($header, ':') !== NULL) {
					list($name, $value) = explode(':', $header, 2);
					if(strtolower(trim($name)) == 'location') {
						return trim($value);
					}
				}
			}
			
			return false;
		}*/
		
		
		
		/**
		 * @param string $message The error message
		 */
		public function __construct($message) {
			$this->message = $message;
		}
		
		
		/**
		 * Overloading method for function calls
		 * 
		 * @param string $name      The name of the function to load
		 * @param array  $arguments An array of arguments to load
		 * 
		 * @return mixed
		 */
		public function __call($funcname, $arguments) {
			if(strpos($funcname, '_') !== NULL) {
				list($action, $name) = explode('_', $funcname, 2);
				if($action == 'tpl') {
					// Initialize error template if not already initialized
					if(!is_a($this->_tpl, $this->tpl)
					&& class_exists($this->tpl)) {
						$this->_tpl = new $this->tpl;
					}
					// Check if method exists and call it if possible
					$method = array($this->_tpl, "get_{$name}");
					if(method_exists($method[0], $method[1])) {
						return call_user_func_array($method, $arguments);
					}
				}
			}
			
			// If the call did not work then show an standard-like error message
			$method = get_class($this) . '::' . $name;
			trigger_error("Call to undefined method {$method}", E_USER_ERROR);
			return null;
		}
		
		
		public function get_full_trace() {
			$trace = array_merge(
				array(
					array(
					  "function" => "__construct",
					  "line"     => $this->getLine(),
					  "file"     => $this->getFile(),
					  "class"    => get_class($this),
					  "object"   => $this,
					  "type"     => "->",
					  "args"     => array()
					)
				),
				parent::getTrace()
			);
			return $trace;
		}
		
		
		/**
		 * Return the body of the page
		 * 
		 * @uses get_message
		 * @uses get_title
		 * @uses get_solutions_html
		 * 
		 * @return string
		 */
		public function get_body() {
			$body = $this->template_body;
			$body = str_replace('$CLASS',    get_class($this),           $body);
			$body = str_replace('$INFO',     $this->get_infos(),         $body);
			$body = str_replace('$MESSAGE',  $this->get_message(),       $body);
			$body = str_replace('$SOLUTIONS',$this->get_solutions_html(),$body);
			$body = str_replace('$TITLE',    $this->get_title(),         $body);
			$body = str_replace('$TRACE',    $this->get_trace_html(),    $body);
			return $body;
		}
		
		
		/**
		 * Return the content of the error page
		 * 
		 * @return string
		 */
		public function get_content() {
			$content = $this->template;
			$content = str_replace('$BODY',    $this->get_body(),    $content);
			$content = str_replace('$CLASS',   get_class($this),     $content);
			$content = str_replace('$MESSAGE', $this->get_message(), $content);
			$content = str_replace('$STYLESHEET', $this->stylesheet, $content);
			$content = str_replace('$TITLE',   $this->get_title(),   $content);
			return $content;
		}
		
		
		/**
		 * Return additional information provided with the error
		 * 
		 * @return string
		 */
		public function get_infos() {
			return $this->infos;
		}
		
		
		/**
		 * Display the content of the error page
		 */
		public function display_content() {
			print($this->get_content());
		}
		
		
		/**
		 * Return an array of headers to send to the client
		 * 
		 * @return array
		 */
		public function get_headers() {
			return (array) $this->headers;
		}
		
		
		/**
		 * Send all headers to the client
		 */
		public function send_headers() {
			if(!headers_sent()) {
				foreach($this->get_headers() as $header) {
					if(substr(trim($header), 0, 5) == 'HTTP/') {
						header($header);
					} else {
						header("HTTP/1.1 $header");
					}
				}
				
				return true;
			} else {
				return false;
			}
		}
		
		
		/**
		 * Return the error message
		 * 
		 * @return string
		 */
		public function get_message() {
			return $this->message;
		}
		
		
		/**
		 * Return an array of possible solutions
		 * 
		 * @return array
		 */
		public function get_solutions() {
			return $this->solutions;
		}
		
		
		/**
		 * Return the list of possible solutions as an unordered HTML list
		 * 
		 * @uses get_solutions
		 * @return string
		 */
		public function get_solutions_html() {
			$html = "<ul>\n";
			foreach($this->get_solutions() as $solution) {
				$html .= "\t<li>$solution</li>\n";
			}
			$html .= "</ul>";
			return $html;
		}
		
		
		public function get_trace() {
			$trace = $this->get_full_trace();
			#$trace = debug_backtrace();
			foreach($trace as $i => $frame) {
				// Add the full function name called to the trace list
				if(isset($frame['class']) && isset($frame['type'])) {
					$trace[$i]['call'] =
					  "{$frame['class']}{$frame['type']}{$frame['function']}";
				} else {
					$trace[$i]['call'] = $frame['function'];
				}
				
				// Add a list of arguments used to the trace list
				$trace[$i]['args'] = '';
				foreach($frame['args'] as $index => $arg) {
					if(is_array($arg)) {
						$trace[$i]['args'] .= 'Array{' . count($arg) . '}';
					} elseif(is_bool($arg)) {
						$trace[$i]['args'] .= $arg ? 'true' : 'false';
					} elseif(is_float($arg) || is_int($arg)) {
						$trace[$i]['args'] .= strval($arg);
					} elseif(is_null($arg)) {
						$trace[$i]['args'] .= 'null';
					} elseif(is_object($arg)) {
						$trace[$i]['args'] .= '<' . get_class($arg) . '>';
					} else {
						$trace[$i]['args'] .= '"' . $arg . '"';
					}
					
					if($index < count($frame['args'])-1) {
						$trace[$i]['args'] .= ', ';
					}
				}
			}
			return $trace;
		}
		
		
		/**
		 * Return an HTML version of the backtrace
		 * 
		 * @return string
		 */
		public function get_trace_html() {
			$bt = $this->get_full_trace();
			#$bt = debug_backtrace();
			
			return self::create_trace_html($bt);
		}
		
		
		/**
		 * Return the error page title
		 * 
		 * @return string
		 */
		public function get_title() {
			$title = $this->title;
			$title = str_replace('$CLASS',   get_class($this),     $title);
			$title = str_replace('$MESSAGE', $this->get_message(), $title);
			return $title;
		}
	}
