<?php
	/**
	 * @package aCMS-AOF
	 */
	require_once(__DIR__ . '/' . "AOFObject.class.php");
	require_once(__DIR__ . '/' . "AOFError.class.php");
	
	
	
	/**
	 * Exception class for a property which is read-only in an AOFData-object
	 */
	class AOFDataPropertyReadOnlyError extends AOFObjectPropertyError {
		/**
		 * @param mixed  $object The object failing to load the method
		 * @param string $name   The name of the property
		 */
		public function __construct($object, $name) {
			parent::__construct($object, $name, "is read-only");
		}
	}
	
	
	
	/**
	 * Base class which allows you to access data in several standard ways
	 * 
	 * All the data to access is stored and is accessible over the get_{$name}()
	 * and set_{$name}() functions. All variables can also be accessed via the
	 * $class->{$name} variable access and the $class[{$name}] array access.
	 * 
	 * Via the `$read_only` attribute you can control if the data variables
	 * should be writeable.
	 */
	abstract class AOFData extends AOFObject implements arrayaccess {
		/**
		 * A list of arguments expected for __construct() which should be
		 * assigned to {@link $data}
		 * 
		 * @see __construct
		 * 
		 * @var array
		 */
		protected $construct_args = array();
		
		/**
		 * Array which stores all data variables
		 * 
		 * @var array
		 */
		protected $data = array();
		
		/**
		 * Controls if the data is read-only or not
		 * 
		 * The {@link $writable} variable will override this value
		 * 
		 * @var bool|array
		 */
		protected $readonly = true;
		
		/**
		 * Controls which variables should always be writable
		 * 
		 * All variable names defined in this array will always be changeable
		 * even if {@link $readonly} is set.
		 * 
		 * @var array
		 */
		protected $writable = array();
		
		
		
		/**
		 * Constructor which loops through the list of elements in
		 * the {@link $construct_args} list
		 * 
		 * The name of the element in the list will be used as variable name and
		 * the argument supplied for construct will be used as variable value.
		 * If the list of supplied arguments is longer than the list of elements
		 * defined in {@link $construct_args} then they will be ignored. If the
		 * list of supplied arguments is shorter than an exception will be
		 * raised.
		 * 
		 * @throws AOFObjectMethodArgumentCountError
		 * 
		 * @param mixed $... The arguments to add to $data
		 */
		public function __construct() {
			// Check if we have enough arguments
			if(func_num_args() < count($this->construct_args)) {
				// Throw the exception
				throw new AOFObjectMethodArgumentCountError($this,
				  '__construct', func_get_args(), count($this->construct_args));
				return false;
			}
			
			$x = 0;
			// Loop through the argument list
			foreach(func_get_args() as $arg) {
				// Skip extra arguments
				if($x > count($this->construct_args)) {
					break;
				// Use the entry in $this->constuct_data as variable name
				// and assign the argument as value
				} else {
					$this->data[$this->construct_args[$x]] = $arg;
				}
				
				$x++;
			}
		}
		
		
		/**
		 * Overloading function for all get_* and set_* function calls
		 * 
		 * If the function name is not get_$name or set_$name then a fatal error
		 * message will be shown. This also applies if the $name part does not
		 * match any defined data variable or if the setter of a variable is
		 * called while in read-only is true.
		 * 
		 * If a valid getter was called then the data variable value will be
		 * returned
		 * 
		 * If a valid setter was called and the variable to set is not read-only
		 * then the value will be changed and true will be returned.
		 * 
		 * If no method could be found then the default action will be called.
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.methods
		 * 
		 * @throws AOFObjectMethodArgumentCountError
		 * 
		 * @param string $funcname  The name of the function to emulate
		 * @param array  $arguments A list of arguments for these functions
		 * 
		 * @return mixed|true
		 */
		public function __call($funcname, $arguments) {
			// Check if $name has at least 2 parts
			if(strpos($funcname, '_') !== NULL) {
				list($action, $name) = explode('_', $funcname, 2);
				
				// If the function call was get_$name and if $name exists then
				// return the data variable of $name
				if($action == 'get' && isset($this->data[$name])) {
					return $this->data[$name];
				}
				
				// Make is_$name like get_{$name} since it is sometimes used for
				// Boolean values but only return true or false
				if($action == 'is' && isset($this->data[$name])) {
					return (bool) $this->data[$name];
				}
				
				// If the function call was set_{$name} and if the variables are
				// not read-only then check if we have enough arguments
				if($action == 'set'
				&&(!$this->readonly || in_array($name, $this->writable))) {
					// If we have enough arguments then change the value
					if(count($arguments) >= 1) {
						$this->data[$name] = $arguments[0];
						return true;
					// Otherwise throw the AOFObject exception
					} else {
						throw new AOFObjectMethodArgumentCountError(
						  $this, $funcname, $arguments, 1);
					}
				}
			}
			
			// Use the default mechanism
			return parent::__call($funcname, $arguments);
		}
		
		
		/**
		 * Overloading function to load the data variables
		 * 
		 * If a data variable cannot be found then the default action will be
		 * called.
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.members
		 * 
		 * @param string $name The name of the data variable to return
		 * 
		 * @return mixed
		 */
		public function __get($name) {
			// Try to load data variable
			if(isset($this->data[$name])) {
				return $this->data[$name];
			// Do default action
			} else {
				return parent::__get($name);
			}
		}
		
		
		/**
		 * Overloading function to check if a data variable exists if it does
		 * not the default action will be called
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.members
		 * 
		 * @param string $name The name of the variable to check
		 * 
		 * @return bool
		 */
		public function __isset($name) {
			if(isset($this->data[$name])) {
				return true;
			} else {
				return parent::__isset($name);
			}
		}
		
		
		/**
		 * Overloading function to change a data variable
		 * 
		 * If the data variable is read-only then an exception will be thrown.
		 * 
		 * If the data variable does not exist then the default action will be
		 * called.
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.member
		 * @throws AOFDataPropertyReadOnlyError
		 * 
		 * @param string $name  The name of the data variable to set
		 * @param mixed  $value The value to change the data variable to
		 */
		public function __set($name, $value) {
			// Check if variable is read-only
			if($this->readonly && !in_array($name, $this->writable)) {
				// Check if property is managed by the AOFData object
				if(isset($this->data[$name])) {
					// Throw an exception
					throw new AOFDataPropertyReadOnlyError($this, $name);
				} else {
					// Set variable in all parents
					parent::__set($name, $value, true);
				}
			// Set variable to new value
			} else {
				// Set item in this class
				$this->data[$name] = $value;
				
				// Set variable in all parents
				parent::__set($name, $value, true);
			}
		}
		
		
		/**
		 * Overloading function to unset a data variable
		 * 
		 * If the data variable is read-only then an exception will be thrown.
		 * 
		 * If the data variable does not exist then the default action will be
		 * called.
		 * 
		 * @link http://php.net/manual/language.oop5.overloading.php#language.oop5.overloading.members
		 * @throws AOFDataPropertyReadOnlyError
		 * @uses AOFObject::__unset
		 * 
		 * @param string $name The name of the data variable to unset
		 * @return bool Success?
		 */
		public function __unset($name) {
			// Check if variable exists
			if(!isset($this->data[$name])) {
				// Do default action
				return parent::__unset($name);
			// Check if the variable is read-only
			} elseif($this->readonly && !in_array($name, $this->writable)) {
				// Check if property is managed by the AOFData object
				if(isset($this->data[$name])) {
					// Throw an exception
					throw new AOFDataPropertyReadOnlyError($this, $name);
					return false;
				} else {
					// Unset variable in all parents
					return parent::__unset($name, true);
				}
			} else {
				// Unset data variable
				unset($this->data[$name]);
				
				// Unset variable in all parents
				return parent::__unset($name, true);
			}
		}
		
		
		public function offsetExists($name) {
			return $this->__isset($name);
		}
		
		public function offsetGet($name) {
			return $this->__get($name);
		}
		
		public function offsetSet($name, $value) {
			return $this->__set($name, $value);
		}
		
		public function offsetUnset($name) {
			return $this->__unset($name);
		}
	}
